<?php

namespace App\Livewire\Doctor\Staff;

use App\Models\User;
use App\Models\Clinic;
use Livewire\Component;
use App\Models\StaffClinic;
use Livewire\WithPagination;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Gate;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class StaffList extends Component
{
    use WithPagination;

    public $searchTerm = '';
    public $statusFilter = 'all';
    public $doctorFilter = 'all';
    public $perPage = 10;
    public $sortField = 'name';
    public $sortDirection = 'asc';
    public $clinic;

    protected $queryString = [
        'search' => ['except' => ''],
        'statusFilter' => ['except' => 'all'],
        'perPage',
        'sortField',
        'sortDirection'
    ];

    public function mount()
    {
        try {
            $this->clinic = Clinic::whereHas('doctorAssociations', function ($query) {
                $query->where('doctor_id', auth()->id())
                    ->where('is_primary', true)
                    ->where('approval_status', 'approved');
            })->firstOrFail();
        } catch (ModelNotFoundException $e) {

            notyf()->error('You do not have a primary clinic assigned. You may need to create one.');
            $this->clinic = null;
        }
    }

    public function updatedSearchTerm()
    {
        $this->resetPage();
    }

    public function applyFilters()
    {
        $this->resetPage();
    }

    public function clearAllFilters()
    {
        $this->statusFilter = 'all';
        $this->doctorFilter = 'all';
        $this->searchTerm = '';
        $this->resetPage();
    }

    public function resetStatusFilter()
    {
        $this->statusFilter = 'all';
        $this->resetPage();
    }
    public function resetDoctorFilter()
    {
        $this->doctorFilter = 'all';
        $this->resetPage();
    }

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
    }

    public function toggleStatus($staffClinic)
    {
        $association = StaffClinic::where('staff_id', $staffClinic['staff_id'])
            ->where('clinic_id', $staffClinic['clinic_id'])
            ->where('doctor_id', $staffClinic['doctor_id'])
            ->firstOrFail();

        $association->status = $association->status === 'active' ? 'inactive' : 'active';
        $association->save();

        notyf()->success('Staff status updated successfully.');
    }

    public function delete($staffClinic)
    {
        try {
            $staffId = $staffClinic['staff_id'];
            if (auth()->id() == $id) {
                notyf()->error('You cannot delete your own account.');
                return;
            }
            
            $hasOtherClinics = StaffClinic::where('staff_id', $staffId)->exists();
            $staffClinic = StaffClinic::where('staff_id', $staffId)
                ->where('clinic_id', $staffClinic['clinic_id'])
                ->where('doctor_id', $staffClinic['doctor_id'])
                ->delete();

            if (!$hasOtherClinics) {
                User::find($staffId)->delete();
            }

            notyf()->success('Staff removed successfully.');
        } catch (\Exception $e) {
            notyf()->error('Failed to delete staff: ' . $e->getMessage());
        }
    }

    public function render()
    {
        $query = $this->clinic->staffAssociations()
            ->with(['staff', 'doctor'])
            ->where('approval_status' , 'approved')
            ->join('users', 'staff_clinics.staff_id', '=', 'users.id')
            ->leftJoin('users as doctors', 'staff_clinics.doctor_id', '=', 'doctors.id'); // Add join for doctors

        if ($this->searchTerm) {
            $query->where(function ($subQuery) {
                $subQuery->where('users.name', 'like', '%' . $this->searchTerm . '%')
                    ->orWhere('users.email', 'like', '%' . $this->searchTerm . '%')
                    ->orWhere('doctors.name', 'like', '%' . $this->searchTerm . '%'); // Add doctor name search
            });
        }

        $doctors = $query->get();

        if ($this->statusFilter !== 'all') {
            $query->where('staff_clinics.status', $this->statusFilter);
        }

        if ($this->doctorFilter !== 'all') {
            $query->where('staff_clinics.doctor_id', $this->doctorFilter);
        }

        switch ($this->sortField) {
            case 'name':
                $query->orderBy('users.name', $this->sortDirection);
                break;
            case 'email':
                $query->orderBy('users.email', $this->sortDirection);
                break;
            case 'status':
                $query->orderBy('staff_clinics.status', $this->sortDirection);
                break;
            default:
                $query->orderBy('users.name', 'asc');
                break;
        }

        $query->select('staff_clinics.*');

        $staffAssociations = $query->paginate($this->perPage);

        return view('livewire.doctor.staff.staff-list', [
            'staffAssociations' => $staffAssociations,
            'allDoctors' => $doctors,
        ]);
    }
}
