<?php

namespace App\Livewire\Doctor\Staff;

use App\Models\User;
use App\Models\Staff;
use Livewire\Component;
use App\Models\StaffClinic;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use App\Models\SubscriptionPlan;
use App\Mail\Staff\CreateStaffMail;
use App\Mail\Staff\UpdateStaffMail;
use App\Mail\Staff\AssignStaffMail;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Hash;
use Livewire\WithPagination;

class StaffForm extends Component
{
    use WithPagination;
    public $clinic;
    public $staffId;
    public $name;
    public $email;
    public $phone_number;
    public $permissions = [];
    public $availablePermissions = [];
    public $permissionGroups = [];
    public $doctors = [];
    public $selectedDoctor;
    public $staffMode = 'create';
    public $existingStaff = [];
    public $selectedExistingStaff;
    public $isClinicOwner = false;
    public $searchTerm = '';
    public $dropdownVisible = false;
    public $profileImage;
    public $currentProfileImage;
    public $selectedClinic;
    public $clinics = [];
    public $status = 'inactive';
    public $password;
    public $password_confirmation;
    public $notes;

    protected function rules()
    {
        $rules = [
            'name' => 'required|string|max:255',
            'status' => 'required|string|max:20',
            'email' => ['required', 'email', Rule::unique('users', 'email')->ignore($this->staffId)],
            'phone_number' => [
                'nullable',
                'string',
                'max:20',
                Rule::unique('users', 'phone_number')->ignore($this->staffId)
            ],
            'password' => 'required|min:6|confirmed',
            'permissions' => 'array|min:1',
            'permissions.*' => 'string|max:3000',
            'selectedDoctor' => 'required|exists:users,id',
            'profileImage' => 'nullable|image|max:2048',
            'notes' => 'nullable|string',
        ];

        if ($this->staffMode === 'select') {
            $rules['selectedExistingStaff'] = 'required|exists:users,id';
            unset($rules['name'], $rules['email'], $rules['phone_number'], $rules['password']);
        }

        if ($this->staffId) {
            $rules = [
                'status' => 'required|string|max:20',
                'permissions' => 'array|min:1',
                'permissions.*' => 'string|max:3000',
            ];
        }

        return $rules;
    }
    public function mount($staff = null, $doctor = null)
    {
        $user = Auth::guard('doctor')->user();

        Log::info('Mounting component for user', [
            'user_id' => $user->id,
            'roles' => $user->getRoleNames(),
            'permissions' => $user->getAllPermissions()->pluck('name'),
        ]);

        $primaryClinic = $user->primaryClinics()->first();

        if (!$primaryClinic) {
            abort(403, 'You do not have any primary clinics to manage staff for.');
        }

        $this->clinic = $primaryClinic;
        $this->isClinicOwner = $user->id === $primaryClinic->pivot->doctor_id;

        if (!$this->isClinicOwner) {
            abort(403, 'You are not authorized to manage staff for this clinic.');
        }

        $this->loadDoctors();
        $this->loadExistingStaff();
        $this->definePermissions();

        if ($staff) {
            $this->selectedDoctor = $doctor->id;
            $this->editStaff($staff);
        }
    }

    protected function loadDoctors()
    {
        $this->doctors = User::whereHas('clinics', function ($query) {
            $query->where('clinic_id', $this->clinic->id);
        })
            ->select('id', 'name')
            ->orderBy('name')
            ->get()
            ->toArray();
    }

    public function resetForm()
    {
        $this->reset([
            'name',
            'email',
            'password',
            'password_confirmation',
            'phone_number',
            'status',
            'notes',
            'staffMode',
            'selectedExistingStaff',
            'searchTerm',
            'dropdownVisible',
            'profileImage',
            'currentProfileImage',
            'permissions'
        ]);
        $this->resetValidation();
    }

    protected function definePermissions()
    {
        $this->availablePermissions = [
            'access admin dashboard',
            'list patients',
            'edit patients',
            'create patients',
            'delete patients',
            'view patient profile',
            'list appointments',
            'create appointments',
            'edit appointments',
            'view schedules',
            'view medical records',
            'view prescriptions',
            'view invoices',
            'process payments',
            // Staff management permissions
            'list staff',
            'create staff',
            'edit staff',
            'view staff',
            'delete staff',
        ];

        $this->permissionGroups = [
            'Dashboard Access' => ['access admin dashboard'],
            'Patient Management' => ['list patients', 'edit patients', 'create patients', 'delete patients', 'view patient profile'],
            'Appointment Management' => ['list appointments', 'create appointments', 'edit appointments'],
            'Medical Records' => ['view schedules', 'view medical records', 'view prescriptions'],
            'Financial' => ['view invoices', 'process payments'],
            'Staff Management' => [
                'list staff',
                'create staff',
                'edit staff',
                'view staff',
                'delete staff'
            ]
        ];

        $this->permissions = ['access admin dashboard'];
    }

    protected function editStaff($staff)
    {
        $this->staffId = $staff->id;
        $staffClinic = StaffClinic::where('staff_id', $staff->id)
            ->where('doctor_id', $this->selectedDoctor)
            ->where('clinic_id', $this->clinic->id)
            ->firstOrFail();

        $this->notes = is_array($staffClinic->notes) ? implode(', ', $staffClinic->notes) : ($staffClinic->notes ?? '');
        $this->status = $staffClinic->status;
        $this->name = $staff->name;
        $this->email = $staff->email;
        $this->phone_number = $staff->phone_number;
        $this->permissions = $staffClinic->permissions ? (is_array($staffClinic->permissions) ? $staffClinic->permissions : json_decode($staffClinic->permissions, true)) : [];
        if ($staff->staff && $staff->staff->profile_image) {
            $this->currentProfileImage = $staff->staff->profile_image;
        }
    }

    public function updatedSearchTerm()
    {
        $this->dropdownVisible = !empty($this->searchTerm);
        $this->loadExistingStaff();
    }

    public function updatedSelectedDoctor()
    {
        $this->loadExistingStaff();
        $this->selectedExistingStaff = null;
    }

    public function loadExistingStaff()
    {
        $query = User::query()
            ->where('id', '!=', auth()->id())
            ->orderBy('name');

        if (!empty($this->searchTerm)) {
            $query->where(function ($q) {
                $q->where('name', 'like', '%' . $this->searchTerm . '%')
                    ->orWhere('email', 'like', '%' . $this->searchTerm . '%');
            });
        }

        $this->existingStaff = $query->take(100)->get();
    }

    public function selectStaff($staffId)
    {
        $this->selectedExistingStaff = $staffId;
        $this->dropdownVisible = false;
        $this->searchTerm = '';
    }

    public function toggleStaffMode($mode)
    {
        $this->resetValidation();
        $this->staffMode = $mode;

        if ($mode === 'create') {
            $this->selectedExistingStaff = null;
        }
    }

    public function updatedSelectedExistingStaff($value)
    {
        if ($value) {
            $staff = User::find($value);
            if ($staff) {
                $this->name = $staff->name;
                $this->email = $staff->email;
                $this->phone_number = $staff->phone_number;
            }
        }
    }

    public function save()
    {
        //dump('Not validate', $this->selectedExistingStaff);

        try {
            $this->validate();
            //dd('Validate');

            if (!$this->isClinicOwner) {
                notyf()->error('Only clinic owners can manage staff.');
                return redirect()->route('doctor.staff.list');
            }

            if (!$this->checkStaffLimits()) {
                return redirect()->back();
            }

            if ($this->staffId) {
                $staff = $this->updateStaff();

                $ownerName = optional($this->clinic->owner)->name ?? 'Clinic Owner';
                $assignedDoctorName = optional(User::find($this->selectedDoctor))->name ?? null;

                $ownerName = $this->formatDoctorName($ownerName);
                $assignedDoctorName = $this->formatDoctorName($assignedDoctorName);

                notyf()->success('Staff updated successfully.');
            } else {
                $staff = $this->createStaff();

                if ($staff) {
                    $ownerName = optional($this->clinic->owner)->name ?? 'Clinic Owner';
                    $ownerName = $this->formatDoctorName($ownerName);

                    notyf()->success('Staff created successfully.');
                    return redirect()->route('doctor.staff.list');
                }

                notyf()->error('Failed to create staff member.');
                return redirect()->back();
            }

            return redirect()->route('doctor.staff.list');
        } catch (\Illuminate\Validation\ValidationException $e) {
            // Log all validation errors
            Log::error('Validation failed', [
                'errors' => $e->validator->errors()->toArray()
            ]);

            // Optional: See exactly which field failed
            foreach ($e->validator->errors()->toArray() as $field => $messages) {
                Log::error("Validation error on '{$field}': " . implode(', ', $messages));
            }

            if ($e->validator->errors()->has('permissions')) {
                notyf()->error('Please select at least one valid permission.');
            }

            return redirect()->back()->withErrors($e->validator)->withInput();
        } catch (\Exception $e) {
            Log::error('Staff management error: ' . $e->getMessage());
            notyf()->error('Operation failed: ' . $e->getMessage());
            return redirect()->back();
        }
    }

    protected function formatDoctorName(?string $name): ?string
    {
        if (!$name)
            return $name;
        return Str::startsWith($name, 'Dr.') ? $name : 'Dr. ' . $name;
    }
    protected function checkStaffLimits()
    {
        if ($this->staffId) {
            return true;
        }

        $subscription = $this->clinic->owner->activeSubscription;

        if (!$subscription) {
            notyf()->error('No active subscription found. Please subscribe to a plan first.');
            return false;
        }

        $plan = $subscription->plan;

        if ($plan->max_staff === null) {
            return true;
        }

        $currentStaffCount = StaffClinic::where('clinic_id', $this->clinic->id)
            ->whereHas('staff', fn($q) => $q->role('doctorstaff'))
            ->where('status', 'active')
            ->count();

        if ($currentStaffCount >= $plan->max_staff) {
            $this->showUpgradeOptions($plan->max_staff);
            notyf()->error("You've reached your staff limit of {$plan->max_staff}.");
            return false;
        }

        return true;
    }

    protected function showUpgradeOptions($currentLimit)
    {
        $nextPlan = SubscriptionPlan::where('max_staff', '>', $currentLimit)
            ->where('plan_status', 'active')
            ->orderBy('max_staff')
            ->first();

        if ($nextPlan) {
            notyf()->error("Upgrade to {$nextPlan->plan_name} to add up to {$nextPlan->max_staff} staff members.");
        }
    }

    protected function updateStaff()
    {
        return DB::transaction(function () {
            $staff = User::findOrFail($this->staffId);

            $staff->syncRoles(['doctorstaff']);
            if (!is_array($this->permissions)) {
                $this->permissions = json_decode($this->permissions, true);
            }
            $staff->syncPermissions($this->permissions);

            $this->updateStaffClinicAssociation($staff);
            return $staff;
        });
    }

    protected function createStaff()
    {
        //dump(['test' , $this->selectedExistingStaff]);
        if ($this->staffMode === 'select') {
            //dd('Select');
            return $this->assignExistingStaff();
        }
        //dd('Create');
        return $this->createNewStaff();
    }

    protected function assignExistingStaff()
    {
        return DB::transaction(function () {
            $staff = User::findOrFail($this->selectedExistingStaff);

            $existingAssignment = StaffClinic::where('staff_id', $staff->id)
                ->where('clinic_id', $this->clinic->id)
                ->where('doctor_id', $this->selectedDoctor)
                ->exists();

            if ($existingAssignment && !$this->staffId) {
                notyf()->error('This user is already assigned to this clinic under the selected doctor.');
                return null;
            }

            if (!$staff->hasRole('doctorstaff')) {
                $staff->assignRole('doctorstaff');
            }

            $this->createStaffClinicAssociation($staff);
            #Mail::to([$staff->email])->send(new AssignStaffMail());
            #Log::info("✅ AssignStaffMail sent to {$staff->email}");

            return $staff;
        });
    }

    protected function createNewStaff()
    {
        return DB::transaction(function () {
            if (User::where('email', $this->email)->exists()) {
                notyf()->error('Email already exists. Please use a different email.');
                return null;
            }

            $staff = User::create([
                'name' => $this->name,
                'email' => $this->email,
                'phone_number' => $this->phone_number,
                'password' => Hash::make($this->password),
                'email_verified_at' => now(),
                'status' => 'active'
            ]);

            $staff->assignRole('doctorstaff');

            $staffData = [
                'user_id' => $staff->id,
                'phone' => $this->phone_number,
                'status' => $this->status,
            ];

            if ($this->profileImage) {
                $path = $this->profileImage->store('staff/profile_images', 'public');
                $staffData['profile_image'] = $path;
            }

            $this->createStaffClinicAssociation($staff);

            Staff::create($staffData);

            return $staff;
        });
    }

    protected function createStaffClinicAssociation(User $staff)
    {
        $this->permissions = array_unique(array_merge($this->permissions, ['access admin dashboard']));

        $association = StaffClinic::create([
            'staff_id' => $staff->id,
            'clinic_id' => $this->clinic->id,
            'doctor_id' => $this->selectedDoctor,
            'added_by' => auth()->id(),
            'status' => 'inactive',
            'approval_status' => 'pendding',
            'permissions' => $this->permissions,
            'notes' => $this->notes ? [$this->notes] : null
        ]);

        if (!$association) {
            throw new \Exception('Failed to create staff clinic association');
        }

        return $association;
    }

    protected function updateStaffClinicAssociation(User $staff)
    {
        $this->permissions = array_unique(array_merge($this->permissions, ['access admin dashboard']));
        $association = StaffClinic::where('staff_id', $staff->id)
            ->where('clinic_id', $this->clinic->id)
            ->where('doctor_id', $this->selectedDoctor)
            ->first();

        if ($association->approval_status === 'approved') {
            $association->update([
                'added_by' => auth()->id(),
                'status' => $this->status,
                'permissions' => json_encode($this->permissions),
                'notes' => $this->notes ? [$this->notes] : null
            ]);
        } else {
            notyf()->error('This Staff has to yet to be Approved by administrator.');
        }

        if (!$association) {
            throw new \Exception('Failed to update staff clinic association');
        }

        return $association;
    }
    public function togglePermission($permission)
    {
        $currentPermissions = is_array($this->permissions) ? $this->permissions : [];

        if (in_array($permission, $currentPermissions)) {
            $this->permissions = array_values(array_diff($currentPermissions, [$permission]));
        } else {
            $this->permissions = array_values(array_unique(array_merge($currentPermissions, [$permission])));
        }

        if (!in_array('access admin dashboard', $this->permissions)) {
            $this->permissions = array_values(array_unique(array_merge($this->permissions, ['access admin dashboard'])));
        }
    }

    public function toggleAllPermissionsInGroup($groupName)
    {
        if (!isset($this->permissionGroups[$groupName])) {
            return;
        }

        $groupPermissions = $this->permissionGroups[$groupName];
        $allCurrentlySelected = count(array_intersect($groupPermissions, $this->permissions)) === count($groupPermissions);

        if ($allCurrentlySelected) {
            // Remove all permissions from this group
            $this->permissions = array_diff($this->permissions, $groupPermissions);
        } else {
            // Add all permissions from this group
            $this->permissions = array_unique(array_merge(
                $this->permissions,
                $groupPermissions
            ));
        }

        $this->resetPage();
    }

    protected function handleProfileImageUpload()
    {
        if ($this->profile_image) {
            try {
                // Store the image in storage/app/public/doctor-profiles directory
                $path = $this->profile_image->store('doctor-profiles', 'public');
                Log::info('Profile image uploaded successfully', ['path' => $path]);
                return $path;
            } catch (\Exception $e) {
                Log::error('Failed to upload profile image: ' . $e->getMessage());
                // Don't throw exception, just return null so doctor creation doesn't fail
                return null;
            }
        }

        return null;
    }

    public function getProfileImageUrl()
    {
        // For new uploads (temporary files)
        if ($this->profile_image && is_object($this->profile_image)) {
            try {
                return $this->profile_image->temporaryUrl();
            } catch (\Exception $e) {
                Log::error('Failed to get temporary URL: ' . $e->getMessage());
                return null;
            }
        }

        // For existing images stored in database
        if ($this->currentProfileImage) {
            // Check if file exists in storage
            if (Storage::disk('public')->exists($this->currentProfileImage)) {
                $url = asset('storage/' . $this->currentProfileImage);
                Log::info('Profile image URL generated', [
                    'path' => $this->currentProfileImage,
                    'url' => $url,
                    'file_exists' => Storage::disk('public')->exists($this->currentProfileImage)
                ]);
                return $url;
            } else {
                Log::warning('Profile image file not found', [
                    'path' => $this->currentProfileImage,
                    'full_path' => storage_path('app/public/' . $this->currentProfileImage)
                ]);
                return null;
            }
        }

        return null;
    }

    public function removeProfileImage()
    {
        if ($this->currentProfileImage && Storage::disk('public')->exists($this->currentProfileImage)) {
            Storage::disk('public')->delete($this->currentProfileImage);
            Log::info('Profile image removed', ['path' => $this->currentProfileImage]);
        }

        $this->currentProfileImage = null;
        $this->profile_image = null;

        // Update database
        if ($this->editId) {
            $user = User::findOrFail($this->editId);
            if ($user->doctor) {
                $user->doctor->update(['profile_image' => null]);
            }
        }

        notyf()->success('Profile image removed successfully.');
    }

    public function render()
    {
        return view('livewire.doctor.staff.staff-form');
    }
}
