<?php

namespace App\Livewire\Doctor\Staff\ClinicSelector;

use Livewire\Component;
use Illuminate\Support\Facades\Log;
use App\Services\DoctorClinicService;

class StaffClinicSelector extends Component
{
    public $selectedClinic;
    public $clinics = [];
    public $isOpen = false;

    public function mount(DoctorClinicService $clinicService)
    {
        $this->clinics = $clinicService->getStaffAssignedClinics(auth()->user());

        $previousClinic = session('current_clinic_id');

        $firstClinic = !empty($this->clinics) ? $this->clinics[0] : null;
        
        if ($previousClinic && collect($this->clinics)->contains('id', $previousClinic)) {
            $this->selectedClinic = $previousClinic;
        } elseif ($firstClinic) {
            $this->selectedClinic = $firstClinic['id'];
            session(['current_clinic_id' => $firstClinic['id']]);
        } else {
            $this->selectedClinic = null;
        }

        if ($this->selectedClinic) {
            $association = $clinicService->getStaffClinicAssociation(auth()->user(), $this->selectedClinic);
            if ($association) {
                session([
                    'clinic_role' => 'staff',
                    'staff_permissions' => $association->permissions,
                    'supervising_doctor_id' => $association->doctor_id
                ]);
            }
        }
    }

    public function selectClinic($clinicId)
    {
        $user = auth()->user();

        Log::channel('permissions')->info('Staff clinic selection changed', [
            'staff_id' => $user->id,
            'staff_name' => $user->name,
            'previous_clinic' => session('current_clinic_id'),
            'new_clinic' => $clinicId
        ]);

        session(['current_clinic_id' => $clinicId]);
        $this->selectedClinic = $clinicId;
        $this->isOpen = false;

        $clinicService = app(DoctorClinicService::class);
        $association = $clinicService->getStaffClinicAssociation($user, $clinicId);

        if ($association) {
            Log::channel('permissions')->info('Staff clinic role assigned', [
                'staff_id' => $user->id,
                'clinic_id' => $clinicId,
                'supervising_doctor_id' => $association->doctor_id,
                'permissions' => $association->permissions
            ]);
            
            session([
                'clinic_role' => 'staff',
                'staff_permissions' => $association->permissions,
                'supervising_doctor_id' => $association->doctor_id
            ]);
        }

        $this->dispatch('clinic-changed-refresh');
    }

    public function render()
    {
        $currentClinic = collect($this->clinics)->firstWhere('id', $this->selectedClinic);

        return view('livewire.doctor.staff.clinic-selector.staff-clinic-selector', [
            'currentClinic' => $currentClinic
        ]);
    }
}