<?php

namespace App\Livewire\Doctor\Profile;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\User;
use App\Models\PatientClinic;
use Illuminate\Support\Facades\Auth;

class DoctorPatientsTable extends Component
{
    use WithPagination;

    public $searchTerm = '';
    public $perPage = 10;
    public $sortField = 'id';
    public $sortDirection = 'desc';
    public $isLoaded = false;

    protected $paginationTheme = 'bootstrap';
    protected $listeners = [
        'confirm-delete-patient' => 'confirmDeletePatient',
    ];

    public function mount()
    {
        $this->isLoaded = true;
    }

    public function updatingSearchTerm()
    {
        $this->resetPage();
    }

    public function getInitials($name)
    {
        if (empty($name)) {
            return 'NA';
        }

        $words = explode(' ', trim($name));

        if (count($words) >= 2) {
            return strtoupper(substr($words[0], 0, 1) . substr($words[count($words) - 1], 0, 1));
        } else {
            return strtoupper(substr($name, 0, 2));
        }
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }

        $this->sortField = $field;
        $this->resetPage();
    }

    public function togglePatientStatus($patientId, $clinicId, $doctorId)
    {
        try {
            $association = PatientClinic::where('patient_id', $patientId)
                ->where('clinic_id', $clinicId)
                ->where('doctor_id', $doctorId)
                ->first();

            if ($association) {
                $association->status = $association->status === 'active' ? 'inactive' : 'active';
                $association->save();

                notyf()->success('Patient status updated successfully.');
            }
        } catch (\Exception $e) {
            notyf()->error('Failed to update patient status.');
        }
    }

       public function confirmDeletePatient($patientId, $clinicId)
    {  
        if (!$patientId || !$clinicId) {
            $this->dispatch('alert', type: 'error', message: 'Invalid patient or clinic.');
            return;
        }

        $this->removePatient($patientId, $clinicId);
        $this->dispatch('hide-delete-modal');
    }
    public function removePatient($patientId, $clinicId)
    {
        try {
            $association = PatientClinic::where('patient_id', $patientId)
                ->where('clinic_id', $clinicId)
                ->where('doctor_id', Auth::id())
                ->first();

            if ($association) {
                $association->delete();
                notyf()->success('Patient removed successfully.');
            }
        } catch (\Exception $e) {
            notyf()->error('Failed to remove patient.');
        }
    }

    public function render()
    {
        $searchTerm = '%' . $this->searchTerm . '%';

        $patients = User::whereHas('roles', function ($q) {
            $q->where('name', 'patient');
        })
            ->whereHas('patientClinics', function ($q) {
                $q->where('doctor_id', Auth::id());
            })
            ->where(function ($q) use ($searchTerm) {
                $q->where('name', 'like', $searchTerm)
                    ->orWhere('email', 'like', $searchTerm);
            })
            ->with([
                'patient',
                'patientClinics' => function ($q) {
                    $q->where('doctor_id', Auth::id())
                        ->with(['clinic', 'doctor']);
                }
            ])
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->perPage);

        return view('livewire.doctor.profile.doctor-patients-table', compact('patients'));
    }
}