<?php
namespace App\Livewire\Doctor\Profile;

use Livewire\Component;
use Livewire\WithFileUploads;
use App\Models\User;
use App\Models\Doctor;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class DoctorBasicInfoComponent extends Component
{
    use WithFileUploads;

    public $name;
    public $phone_number;
    public $specialization;
    public $years_of_experience;
    public $license_number;
    public $bio;
    public $existing_profile_image;
    public $profile_image;

    protected $rules = [
        'name' => 'required|string|max:255',
        'phone_number' => 'required|string|max:20',
        'specialization' => 'required|string|max:255',
        'years_of_experience' => 'nullable|numeric|min:0',
        'license_number' => 'nullable|string|max:255',
        'bio' => 'nullable|string|max:2000',
        'profile_image' => 'nullable|image|max:2048',
    ];

    public function mount()
    {
        $user = Auth::user();
        $doctorProfile = $user->doctorProfile;
        $doctor = $user->doctor;

        $this->name = $user->name;
        $this->phone_number = $user->phone_number;
        $this->specialization = $doctorProfile->specialization ?? '';
        $this->years_of_experience = $doctorProfile->years_of_experience ?? 0;
        $this->license_number = $doctorProfile->license_number ?? '';
        $this->bio = $doctorProfile->bio ?? '';
        $this->existing_profile_image = $doctor->profile_image ?? '';
    }

    public function save()
    {
        $this->validate();

        $user = Auth::user();
        $doctorProfile = $user->doctorProfile;

        $user->update([
            'name' => $this->name,
            'phone_number' => $this->phone_number,
        ]);

        $profileImagePath = $this->handleProfileImageUpdate($user);

        Doctor::updateOrCreate(
            ['user_id' => $user->id],
            [
                'specialization' => $this->specialization,
                'experience_years' => $this->years_of_experience,
                'license_number' => $this->license_number,
                'bio' => $this->bio,
                'profile_image' => $profileImagePath,
            ]
        );

        $doctorProfile->update([
            'specialization' => $this->specialization,
            'years_of_experience' => $this->years_of_experience,
            'license_number' => $this->license_number,
            'bio' => $this->bio,
            'profile_image' => $profileImagePath,
        ]);

        notyf()->success('Profile updated successfully.');
    }

    protected function handleProfileImageUpdate(User $user)
    {
        if ($this->profile_image) {
            if ($this->existing_profile_image && Storage::disk('public')->exists($this->existing_profile_image)) {
                Storage::disk('public')->delete($this->existing_profile_image);
                Log::info('Old profile image deleted', ['path' => $this->existing_profile_image]);
            }

            try {
                $path = $this->profile_image->store('doctor-profiles', 'public');
                Log::info('Profile image updated successfully', ['path' => $path]);
                return $path;
            } catch (\Exception $e) {
                Log::error('Failed to update profile image: ' . $e->getMessage());
                return $this->existing_profile_image;
            }
        }

        return $this->existing_profile_image;
    }

    public function removeProfileImage()
    {
        if ($this->existing_profile_image && Storage::disk('public')->exists($this->existing_profile_image)) {
            Storage::disk('public')->delete($this->existing_profile_image);
            Log::info('Profile image removed', ['path' => $this->existing_profile_image]);
        }

        $this->existing_profile_image = null;
        $this->profile_image = null;

        if ($this->editId) {
            $user = User::findOrFail($this->editId);
            if ($user->doctor) {
                $user->doctor->update(['profile_image' => null]);
            }
        }

        notyf()->success('Profile image removed successfully.');
    }

    public function getProfileImageUrl()
    {
        if ($this->profile_image && is_object($this->profile_image)) {
            try {
                return $this->profile_image->temporaryUrl();
            } catch (\Exception $e) {
                Log::error('Failed to get temporary URL: ' . $e->getMessage());
                return null;
            }
        }

        if ($this->existing_profile_image) {
            if (Storage::disk('public')->exists($this->existing_profile_image)) {
                $url = asset('storage/' . $this->existing_profile_image);
                Log::info('Profile image URL generated', [
                    'path' => $this->existing_profile_image,
                    'url' => $url,
                    'file_exists' => Storage::disk('public')->exists($this->existing_profile_image)
                ]);
                return $url;
            } else {
                Log::warning('Profile image file not found', [
                    'path' => $this->existing_profile_image,
                    'full_path' => storage_path('app/public/' . $this->existing_profile_image)
                ]);
                return null;
            }
        }

        return null;
    }

    public function render()
    {
        return view('livewire.doctor.profile.doctor-basic-info-component');
    }
}