<?php

namespace App\Livewire\Doctor\Patient;

use App\Models\PatientClinic;
use App\Models\User;
use Illuminate\Support\Facades\Gate;
use Livewire\Component;
use Livewire\WithPagination;

class PatientList extends Component
{
    use WithPagination;

    public $searchTerm = '';
    public $statusFilter = 'all';
    public $genderFilter = 'all';
    public $doctorFilter = 'all';
    public $perPage = 10;
    public $sortBy = 'recent';

    protected $queryString = [
        'searchTerm' => ['except' => ''],
        'statusFilter' => ['except' => 'all'],
        'genderFilter' => ['except' => 'all'],
        'doctorFilter' => ['except' => 'all'],
        'perPage' => ['except' => 10],
        'sortBy' => ['except' => 'recent'],
    ];

    public function render()
    {
        $searchTerm = '%' . $this->searchTerm . '%';

        $patients = PatientClinic::with([
            'patient',
            'doctor.doctorProfile'
        ])->whereHas('patient', function ($query) {
                $query->where('status', 'active')
                ->whereHas('patient', function ($query) {
                    $query->where('status', 'active');
                });
            })
            ->where('clinic_id', session('current_clinic_id'))
            ->join('users as patient_users', 'patient_clinics.patient_id', '=', 'patient_users.id')
            ->leftJoin('users as doctor_users', 'patient_clinics.doctor_id', '=', 'doctor_users.id')
            ->where(function ($q) use ($searchTerm) {
                $q->where('patient_users.name', 'like', $searchTerm)
                    ->orWhere('patient_users.email', 'like', $searchTerm);
            })
            ->when($this->statusFilter !== 'all', function ($q) {
                $q->where('patient_clinics.status', $this->statusFilter);
            })
            ->when($this->genderFilter !== 'all', function ($q) {
                $q->whereHas('patient', function ($q) {
                    $q->where('gender', $this->genderFilter);
                });
            })
            ->when($this->doctorFilter !== 'all', function ($q) {
                $q->where('patient_clinics.doctor_id', $this->doctorFilter)
                    ->where('patient_clinics.status', 'active');
            })
            ->when($this->sortBy === 'recent', function ($q) {
                $q->orderBy('patient_clinics.created_at', 'desc');
            })
            ->when($this->sortBy === 'oldest', function ($q) {
                $q->orderBy('patient_clinics.created_at', 'asc');
            })
            ->select('patient_clinics.*')
            ->paginate($this->perPage);

        $doctors = User::whereHas('roles', function ($q) {
            $q->where('group', 'doctor_group')
                ->where('name', '!=', 'doctorstaff');
        })
            ->with('doctorProfile')
            ->get(['id', 'name']);

        return view('livewire.doctor.patient.patient-list', [
            'patients' => $patients,
            'totalPatients' => $patients->total(),
            'doctors' => $doctors
        ]);
    }

    public function resetFilters()
    {
        $this->reset([
            'statusFilter',
            'genderFilter',
            'doctorFilter'
        ]);
    }

    public function toggleStatus($patientClinic)
    {
        //dd($patientClinic);
        Gate::authorize('edit patients');
        $patientClinic = PatientClinic::where('patient_id', $patientClinic['patient_id'])
            ->where('clinic_id', $patientClinic['clinic_id'])
            ->where('doctor_id', $patientClinic['doctor_id'])
            ->first();
        $patientClinic->status = $patientClinic->status === 'active' ? 'inactive' : 'active';
        $patientClinic->save();

        notyf()->success('Patient status updated successfully.');
    }

    public function deletePatient($id)
    {
        if (auth()->id() == $id) {
            notyf()->error('You cannot delete your own account.');
            return;
        }

        $user = User::findOrFail($id);

        if ($user->patient) {
            $user->patient->delete();
        }

        $user->delete();

        notyf()->success('Patient deleted successfully.');
    }

    public function sortBy($type)
    {
        $this->sortBy = $type;
        $this->resetPage();
    }
}
