<?php

namespace App\Livewire\Doctor\Doctor;

use App\Models\User;
use App\Models\Clinic;
use App\Models\Doctor;
use App\Models\DoctorProfile;
use Livewire\Component;
use App\Models\DoctorClinic;
use App\Mail\Doctor\CreateDoctorMail;
use App\Mail\Doctor\UpdateDoctorMail;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Livewire\Attributes\Computed;
use Livewire\Features\SupportFileUploads\WithFileUploads;
use Livewire\WithPagination;

class DoctorForm extends Component
{
    use WithPagination , WithFileUploads;

    public $name, $email, $temporaryPassword;
    public $specialization, $qualifications, $experience_years, $license_number, $bio;
    public $status = 'inactive';
    public $editId = null;
    public $clinic;
    public $selected_clinics = [];
    public $primary_clinic_id = null;
    public $isEmbedded = false;
    public $clinicForm = false;
    public $has_primary_clinic = false;
    public $clinicSearch = '';
    public $showAllClinics = true;
    
    // New properties for existing doctor selection
    public $doctorMode = 'create';
    public $existingDoctors = [];
    public $selectedExistingDoctor;
    public $searchTerm = '';
    public $dropdownVisible = false;
    public $availablePermissions = [];
    public $permissionGroups = [];
    public $permissions = [];
    public $existing_profile_image;
    public $profile_image;

    protected $rules = [
        'name' => 'required|string|max:255',
        'email' => 'required|email|max:255|unique:users,email',
        'specialization' => 'required|string|max:255',
        'qualifications' => 'nullable|string',
        'experience_years' => 'nullable|integer|min:0',
        'license_number' => 'required|string|max:255|unique:doctors,license_number',
        'bio' => 'nullable|string',
        'status' => 'required|in:active,inactive',
        'selected_clinics' => 'nullable|array',
        'selected_clinics.*' => 'exists:clinics,id',
        'permissions' => 'array',
        'profile_image' => 'nullable|image|max:10240', // max 10MB
    ];

    public function mount($editId = null, $isEmbedded = false, $clinicForm = false)
    {
        $this->authorizeUser($editId);
        $this->initializeProperties($editId, $isEmbedded, $clinicForm);
        $this->definePermissions();
        $this->clinic = session('current_clinic_id');
        $this->loadExistingDoctors();
        $this->permissions = [];

        if ($editId) {
            $this->loadDoctorData($editId);
        } else {
            Log::info('Creating new doctor form initialized.');
        }
    }

    protected function authorizeUser($editId)
    {
        if ($editId) {
            Gate::authorize('edit doctors');
        } else {
            Gate::authorize('create doctors');
        }
    }

    protected function initializeProperties($editId, $isEmbedded, $clinicForm)
    {
        $this->isEmbedded = $isEmbedded;
        $this->clinicForm = $clinicForm;
        $this->editId = $editId;
    }

    protected function definePermissions()
    {
        $this->availablePermissions = [
            'access admin dashboard',
            'list patients',
            'view patient profile',
            'list appointments',
            'create appointments',
            'edit appointments',
            'view schedules',
            'view medical records',
            'view prescriptions',
            'view invoices',
            'process basic payments',
            // Staff management permissions
            'list staff',
            'create staff',
            'edit staff',
            'view staff',
            'delete staff',
        ];

        $this->permissionGroups = [
            'Dashboard Access' => ['access admin dashboard'],
            'Patient Management' => ['list patients', 'view patient profile'],
            'Appointment Management' => ['list appointments', 'create appointments', 'edit appointments'],
            'Medical Records' => ['view schedules', 'view medical records', 'view prescriptions'],
            'Financial' => ['view invoices', 'process basic payments'],
            'Staff Management' => [
                'list staff', 'create staff', 'edit staff', 'view staff', 'delete staff'
            ]
        ];
    }

    public function toggleAllPermissionsInGroup($groupName)
    {
        if (!isset($this->permissionGroups[$groupName])) {
            return;
        }

        $groupPermissions = $this->permissionGroups[$groupName];
        $allCurrentlySelected = count(array_intersect($groupPermissions, $this->permissions)) === count($groupPermissions);

        if ($allCurrentlySelected) {
            // Remove all permissions from this group
            $this->permissions = array_diff($this->permissions, $groupPermissions);
        } else {
            // Add all permissions from this group
            $this->permissions = array_unique(array_merge(
                $this->permissions,
                $groupPermissions
            ));
        }

        $this->resetPage();
    }

    public function loadExistingDoctors()
    {
        $clinic = Clinic::find($this->clinic);
        $query = User::whereHas('roles', function ($q) {
            $q->whereIn('name', ['doctor', 'primary_doctor', 'sub_doctor']);
        })
            ->where('status', 'active')
            ->orderBy('name');

        // Exclude current clinic's primary doctor if exists
        if ($clinic->primaryDoctor) {
            $query->where('id', '!=', $clinic->primaryDoctor->doctor_id);
        }

        // Exclude doctors already associated with this clinic
        $existingDoctorIds = $clinic->doctors()->pluck('users.id')->toArray();
        if (!empty($existingDoctorIds)) {
            $query->whereNotIn('id', $existingDoctorIds);
        }

        // Search functionality
        if (!empty($this->searchTerm)) {
            $query->where(function ($q) {
                $q->where('name', 'like', '%' . $this->searchTerm . '%')
                    ->orWhere('email', 'like', '%' . $this->searchTerm . '%');
            });
        }

        $this->existingDoctors = $query->take(100)->get();

    }

    public function updatedSearchTerm()
    {
        $this->dropdownVisible = !empty($this->searchTerm);
        $this->loadExistingDoctors();
    }

    public function selectDoctor($doctorId)
    {
        $this->selectedExistingDoctor = $doctorId;
        $this->dropdownVisible = false;
        $this->searchTerm = '';
    }

    public function toggleDoctorMode($mode)
    {
        $this->resetValidation();
        $this->doctorMode = $mode;

        if ($mode === 'create') {
            $this->selectedExistingDoctor = null;
        }
    }

    public function updatedSelectedExistingDoctor($value)
    {
        if ($value) {
            $doctor = User::with('doctor')->find($value);
            if ($doctor) {
                $this->name = $doctor->name;
                $this->email = $doctor->email;
                
                if ($doctor->doctor) {
                    $this->specialization = $doctor->doctor->specialization;
                    $this->qualifications = $doctor->doctor->qualifications;
                    $this->experience_years = $doctor->doctor->experience_years;
                    $this->license_number = $doctor->doctor->license_number;
                    $this->bio = $doctor->doctor->bio;
                }
            }
        }
    }

    protected function loadDoctorData($editId)
    {
        try {
            $user = User::with(['doctor', 'roles', 'clinics'])->findOrFail($editId);
            $this->existing_profile_image = $user->doctor->profile_image;
            Log::info('Editing doctor form initialized.', ['user_id' => $editId]);

            $this->fillUserData($user);
            $this->fillPermissionsData($user);
        } catch (\Exception $e) {
            Log::error('Failed to load doctor for editing: ' . $e->getMessage(), ['user_id' => $editId]);
            notyf()->error('Failed to load doctor details.');
        }
    }

    protected function fillUserData(User $user)
    {
        $this->name = $user->name;
        $this->email = $user->email;
        $this->status = $user->status;
    }

    protected function fillPermissionsData(User $user)
    {
        $doctorClinic = DoctorClinic::where('doctor_id', $user->id)
            ->where('clinic_id', $this->clinic)
            ->where('is_primary' , 0)
            ->firstOrFail();
        $this->permissions = $doctorClinic->extra_permissions ? (is_array($doctorClinic->extra_permissions) ? $doctorClinic->extra_permissions:json_decode($doctorClinic->extra_permissions , true)) : [];

    }

    #[Computed]
    public function clinics()
    {
        return Clinic::where('status', 'active')
            ->orderBy('name')
            ->get();
    }

    #[Computed]
    public function filteredClinics()
    {
        $clinics = collect();

        // 1. Add current authenticated user's primary clinic (for create form)
        if (!$this->editId && auth()->user()->hasPrimaryClinic()) {
            $primaryClinic = DoctorClinic::where('doctor_id', auth()->id())
                ->where('is_primary', true)
                ->where('approval_status', 'approved')
                ->first();

            if ($primaryClinic) {
                $clinics->push(Clinic::find($primaryClinic->clinic_id));
            }
        }

        // 2. Add edited doctor's primary clinic (for edit form)
        if ($this->editId) {
            $doctorPrimaryClinic = DoctorClinic::where('doctor_id', $this->editId)
                ->where('is_primary', true)
                ->where('approval_status', 'approved')
                ->first();

            if ($doctorPrimaryClinic) {
                $clinics->push(Clinic::find($doctorPrimaryClinic->clinic_id));

                // Also add the authenticated user's primary clinic if different
                if (auth()->user()->hasPrimaryClinic()) {
                    $authPrimaryClinic = DoctorClinic::where('doctor_id', auth()->id())
                        ->where('is_primary', true)
                        ->where('approval_status', 'approved')
                        ->first();

                    if ($authPrimaryClinic && $authPrimaryClinic->clinic_id != $doctorPrimaryClinic->clinic_id) {
                        $clinics->push(Clinic::find($authPrimaryClinic->clinic_id));
                    }
                }
            }
        }

        return $clinics->unique()->sortBy('name');
    }

    public function createDoctor()
    {
        Gate::authorize('create doctors');
        
        // Update validation rules based on mode
        $rules = $this->rules;
        if ($this->doctorMode === 'select') {
            $rules['selectedExistingDoctor'] = 'required|exists:users,id';
            unset($rules['name'], $rules['email'], $rules['password'], $rules['specialization'], $rules['license_number']);
        }
        
        $this->validate($rules);
        if(!$this->checkDoctorLimits()){
            return;
        };

        try {
            DB::transaction(function () {
                if ($this->doctorMode === 'select') {
                    $user = $this->assignExistingDoctor();
                } else {
                    $user = $this->createUser();
                    $this->assignDoctorRole($user);
                    $this->createDoctorRecord($user);
                }
                
                $this->createDoctorClinicAssociation($user);

                $this->handleSuccessfulCreation($user);

                #Mail::to($user->email)->send(new CreateDoctorMail());
                #Log::info("✅ CreateDoctorMail sent to {$user->email}");
            });
        } catch (\Exception $e) {
            Log::error('Doctor creation failed: ' . $e->getMessage());
            notyf()->error( 'Failed to create doctor: ' . $e->getMessage());
        }
    }

    protected function checkDoctorLimits()
    {
        if ($this->editId) {
            return true;
        }

        $selectedClinic = Clinic::findOrFail($this->clinic);
        if (!$selectedClinic->activeSubscription()) {
            notyf()->error( 'Failed to create doctor: No Active Subscription Plan.');
            return false;
        } else {
            $activeSubscription = $selectedClinic->activeSubscription();
            $subscriptionPlan = $activeSubscription->plan;
            $totalDoctorsUnderClinic = DoctorClinic::where('clinic_id', $this->clinic)->where('status' , 'active')->count();
            
            // Check if current doctor count is below the maximum allowed
            if ($totalDoctorsUnderClinic >= $subscriptionPlan->max_doctors) {
                notyf()->error( 'Failed to create doctor: You’ve reached the maximum doctors allowed under your current plan. Upgrade now to add more.');
                notyf()->error("You've reached your doctor limit of {$subscriptionPlan->max_doctors}.");
                return false;
            }
        } 

        return true;
    }

    protected function assignExistingDoctor()
    {
        $user = User::findOrFail($this->selectedExistingDoctor);
        
        if (!$user->hasRole('primary_doctor')) {
            $this->assignDoctorRole($user);
        }
        
        return $user;
    }

    protected function createUser()
    {
        $this->temporaryPassword = "admin@123";
        return User::create([
            'name' => $this->name,
            'email' => $this->email,
            'password' => Hash::make($this->temporaryPassword),
            'status' => $this->status,
        ]);
    }

    protected function assignDoctorRole(User $user)
    {
        $doctorRole = Role::where('name', 'doctor')
            ->where('guard_name', 'doctor')
            ->firstOrFail();
        $user->assignRole($doctorRole);
    }

    protected function createDoctorRecord(User $user)
    {
        $profileImagePath = $this->handleProfileImageUpload();

        Doctor::create([
            'user_id' => $user->id,
            'specialization' => $this->specialization,
            'qualifications' => $this->qualifications,
            'experience_years' => $this->experience_years,
            'license_number' => $this->license_number,
            'bio' => $this->bio,
            'profile_image' => $profileImagePath,
        ]);       

        DoctorProfile::create([
            'user_id' => $user->id,
            'specialization' => $this->specialization,
            'years_of_experience' => $this->experience_years,
            'license_number' => $this->license_number,
            'bio' => $this->bio,
            'verification_status' => 'verified',
        ]);
    }

    protected function handleProfileImageUpload()
    {
        if ($this->profile_image) {
            try {
                // Store the image in storage/app/public/doctor-profiles directory
                $path = $this->profile_image->store('doctor-profiles', 'public');
                Log::info('Profile image uploaded successfully', ['path' => $path]);
                return $path;
            } catch (\Exception $e) {
                Log::error('Failed to upload profile image: ' . $e->getMessage());
                // Don't throw exception, just return null so doctor creation doesn't fail
                return null;
            }
        }

        return null;
    }

    protected function handleSuccessfulCreation(User $user)
    {
        if ($this->isEmbedded && $this->clinicForm) {
            $this->dispatch('doctorCreated', $user->id);
            session()->flash('message', 'Doctor created successfully.');
            return;
        }

        session()->flash('message', 'Doctor created successfully.');
        $this->redirect(route('doctor.doctor-management.list'));
    }

    public function updateDoctor()
    {
        Gate::authorize('edit doctors');
        $this->validate($this->getUpdateRules());

        try {
            DB::transaction(function () {
                $user = User::findOrFail($this->editId);
                $this->updateDoctorClinicAssociation($user);

                #Mail::to($user->email)->send(new UpdateDoctorMail());
                #Log::info("✅ UpdateDoctorMail sent to {$user->email}");
            });

            notyf()->success('Doctor updated successfully.');
            return redirect()->route('doctor.doctor-management.edit', ['id' => $this->editId]);
        } catch (\Exception $e) {
            Log::error('Doctor update failed: ' . $e->getMessage());
            notyf()->error('Failed to update doctor: ' . $e->getMessage());
        }
    }

    protected function getUpdateRules()
    {
        $rules = [
            'permissions' => 'array',
            'permissions.*' => 'string|max:3000',
        ];

        return $rules;
    }

    public function getPrimaryClinicName()
    {
        if (!$this->primary_clinic_id) {
            return '';
        }

        $clinic = $this->clinics->where('id', $this->primary_clinic_id)->first();
        return $clinic ? $clinic->name : '';
    }

    public function updatedSelectedClinics($value)
    {
        // Ensure primary clinic is always selected
        if (
            $this->has_primary_clinic &&
            !empty($this->primary_clinic_id) &&
            !in_array($this->primary_clinic_id, $this->selected_clinics)
        ) {
            $this->selected_clinics[] = $this->primary_clinic_id;
        }
    }

    protected function createDoctorClinicAssociation(User $doctor)
    {
        $this->permissions = is_array($this->permissions) ? $this->permissions : [];
        $this->permissions = array_unique(array_merge($this->permissions, ['access admin dashboard']));

        $association = DoctorClinic::create([
            'doctor_id' => $doctor->id,
            'clinic_id' => $this->clinic,
            'added_by' => auth()->id(),
            'status' => 'inactive',
            'approval_status' => 'pending',
            #'approved_at' => now(),
            'is_primary' => false,
            'in_person_duration' => 15,
            'video_duration' => 15,
            'home_visit_duration' => 30,
            'offers_in_person' => true,
            'offers_video' => true,
            'offers_home_visit' => false,
            'override_default_permissions' => true,
            'extra_permissions' => json_encode($this->permissions)
        ]);

        if (!$association) {
            throw new \Exception('Failed to create doctor clinic association');
        }

        return $association;
    }

    protected function updateDoctorClinicAssociation(User $doctor)
    {
        $this->permissions = array_unique(array_merge($this->permissions, ['access admin dashboard']));

        $association = DoctorClinic::updateOrCreate(
            [
                'doctor_id' => $doctor->id,
                'clinic_id' => $this->clinic
            ],
            [
                'is_primary' => false,
                'in_person_duration' => 15,
                'video_duration' => 15,
                'home_visit_duration' => 30,
                'offers_in_person' => true,
                'offers_video' => true,
                'offers_home_visit' => false,
                'override_default_permissions' => true,
                'extra_permissions' => json_encode($this->permissions)
            ]
        );

        if (!$association) {
            throw new \Exception('Failed to update doctor clinic association');
        }

        return $association;
    }

    public function togglePermission($permission)
    {
        $currentPermissions = is_array($this->permissions) ? $this->permissions : [];

        if (in_array($permission, $currentPermissions)) {
            $this->permissions = array_values(array_diff($currentPermissions, [$permission]));
        } else {
            $this->permissions = array_values(array_unique(array_merge($currentPermissions, [$permission])));
        }

        if (!in_array('access admin dashboard', $this->permissions)) {
            $this->permissions = array_values(array_unique(array_merge($this->permissions, ['access admin dashboard'])));
        }
        #dump($this->permissions);

    }

    #[Computed]
    public function selectedClinicsNames()
    {
        $selectedNames = [];
        foreach ($this->selected_clinics as $id) {
            $clinic = $this->clinics->firstWhere('id', $id);
            if ($clinic) {
                $name = $clinic->name;
                if ($this->has_primary_clinic && $id === $this->primary_clinic_id) {
                    $name .= ' (Primary)';
                }
                $selectedNames[] = $name;
            }
        }

        return $selectedNames;
    }

    public function removeProfileImage()
    {
        if ($this->existing_profile_image && Storage::disk('public')->exists($this->existing_profile_image)) {
            Storage::disk('public')->delete($this->existing_profile_image);
            Log::info('Profile image removed', ['path' => $this->existing_profile_image]);
        }

        $this->existing_profile_image = null;
        $this->profile_image = null;

        // Update database
        if ($this->editId) {
            $user = User::findOrFail($this->editId);
            if ($user->doctor) {
                $user->doctor->update(['profile_image' => null]);
            }
        }

        notyf()->success('Profile image removed successfully.');
    }

    public function getProfileImageUrl()
    {
        // For new uploads (temporary files)
        if ($this->profile_image && is_object($this->profile_image)) {
            try {
                return $this->profile_image->temporaryUrl();
            } catch (\Exception $e) {
                Log::error('Failed to get temporary URL: ' . $e->getMessage());
                return null;
            }
        }

        // For existing images stored in database
        if ($this->existing_profile_image) {
            // Check if file exists in storage
            if (Storage::disk('public')->exists($this->existing_profile_image)) {
                $url = asset('storage/' . $this->existing_profile_image);
                Log::info('Profile image URL generated', [
                    'path' => $this->existing_profile_image,
                    'url' => $url,
                    'file_exists' => Storage::disk('public')->exists($this->existing_profile_image)
                ]);
                return $url;
            } else {
                Log::warning('Profile image file not found', [
                    'path' => $this->existing_profile_image,
                    'full_path' => storage_path('app/public/' . $this->existing_profile_image)
                ]);
                return null;
            }
        }

        return null;
    }

    public function render()
    {
        return view('livewire.doctor.doctor.doctor-form');
    }

}