<?php

namespace App\Livewire\Doctor\Auth;

use App\Models\User;
use Livewire\Component;
use App\Models\ActiveDevice;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;
use Illuminate\Validation\ValidationException;

class LoginForm extends Component
{
    public string $email = '';
    public string $mobile = '';
    public string $password = '';
    public string $role = '';
    public bool $remember = false;
    public bool $useMobile = false;

    public function mount($role)
    {
        $this->role = $role;
    }

    protected function getDeviceRole(): string
    {
        return match ($this->role) {
            'doctor' => 'doctor',
            'doctor_staff' => 'staff',
            default => 'user',
        };
    }

    protected function generateDeviceId($user): string
    {
        return hash('sha256', implode('|', [
            $user->id,
            request()->ip(),
            microtime(),
            bin2hex(random_bytes(16))
        ]));
    }

    public function toggleLoginMethod()
    {
        $this->useMobile = !$this->useMobile;
        $this->resetValidation();
    }

    public function login()
    {
        foreach (['web', 'superadmin', 'doctor', 'staff', 'patient'] as $guard) {
            if (Auth::guard($guard)->check()) {
                Auth::guard($guard)->logout();
                Session::flush();
                Session::regenerate();
                break;
            }
        }
        if ($this->useMobile) {
            $this->validate([
                'mobile' => 'required|string|min:10',
                'password' => 'required',
            ]);
        } else {
            $this->validate([
                'email' => 'required|email',
                'password' => 'required',
            ]);
        }

        Log::info('Doctor login attempt', [
            'login_method' => $this->useMobile ? 'mobile' : 'email',
            'identifier' => $this->useMobile ? $this->mobile : $this->email,
            'requested_role' => $this->role
        ]);

        $credentials = [
            'password' => $this->password,
            $this->useMobile ? 'mobile' : 'email' => $this->useMobile ? $this->mobile : $this->email
        ];

        if (!Auth::guard('doctor')->attempt($credentials, $this->remember)) {
            Log::warning('Doctor authentication failed', [
                'identifier' => $this->useMobile ? $this->mobile : $this->email
            ]);
            throw ValidationException::withMessages([
                $this->useMobile ? 'mobile' : 'email' => 'Invalid credentials'
            ]);
        }

        $user = Auth::guard('doctor')->user();

        Log::info('Doctor user authenticated', [
            'user_id' => $user->id,
            'user_email' => $user->email,
            'user_mobile' => $user->mobile,
            'user_roles' => $user->roles->pluck('name')->toArray(),
            'requested_role' => $this->role
        ]);

        // Check role access based on URL
        if (!$this->validateDoctorRole($user)) {
            Auth::guard('doctor')->logout();
            throw ValidationException::withMessages([
                $this->useMobile ? 'mobile' : 'email' => 'Access denied'
            ]);
        }

        if(!$this->checkAuthRequirements($user)){
            return;
        }
        

        $deviceId = $this->generateDeviceId($user);
        $deviceRole = $this->getDeviceRole(); // Now returns shorter codes
        $sessionId = Session::getId();
        $deviceName = $this->getDeviceName(request()->userAgent());

        Session::put('device_id', $deviceId);
        Session::put('current_role', $deviceRole);
        Session::put('session_id', $sessionId);

        Log::info('Setting active doctor device', [
            'user_id' => $user->id,
            'device_role' => $deviceRole,
            'session_role' => Session::get('current_role'),
            'device_name' => $deviceName
        ]);

        try {
            ActiveDevice::updateOrCreate(
                [
                    'user_id' => $user->id,
                    'device_id' => $deviceId,
                    'role_type' => $deviceRole,
                ],
                [
                    'platform' => 'web',
                    'device_name' => $deviceName,
                    'ip_address' => request()->ip(),
                    'login_method' => $this->useMobile ? 'mobile' : 'email',
                    'is_active' => true,
                    'login_at' => now(),
                    'last_active_at' => now(),
                    'session_id' => $sessionId,
                ]
            );
        } catch (\Exception $e) {
            Log::error('Failed to update active device', [
                'error' => $e->getMessage(),
                'user_id' => $user->id,
                'device_role' => $deviceRole
            ]);
            // Continue with login even if device tracking fails
        }

        return redirect()->intended($this->getDashboardRoute());
    }
    protected function validateDoctorRole(User $user)
    {
        if ($this->role === 'doctor' && !$user->hasRole(['doctor', 'primary_doctor', 'sub_doctor'])) {
            return false;
        }

        if ($this->role === 'doctor_staff' && !$user->hasRole('doctorstaff')) {
            return false;
        }

        return true;
    }

    protected function checkAuthRequirements($user)
    {
        if ($user->status === "inactive") {
            Auth::guard('doctor')->logout();
            notyf()->error('Doctor Authentication Failed Inactive Doctor.');
            return false;
        }
        if ($user->hasRole(['doctor', 'primary_doctor', 'sub_doctor']) && $user->activeClinicAssociations() <= 0) {
            Auth::guard('doctor')->logout();
            notyf()->error('Doctor Authentication Failed No Active Clinic Association.');
            return false;
        }
        if ($user->hasRole(['doctor', 'primary_doctor', 'sub_doctor']) && !$user->doctor) {
            Auth::guard('doctor')->logout();
            notyf()->error('Doctor Authentication Failed User Is Not A Doctor.');
            return false;
        }
        if ($user->hasRole(['doctorstaff']) && $user->activeStaffClinicAssociations() <= 0) {
            Auth::guard('doctor')->logout();
            notyf()->error('Doctor Staff Authentication Failed No Active Clinic Association.');
            return false;
        }
        if ($user->hasRole(['doctorstaff']) && !$user->staff) {
            Auth::guard('doctor')->logout();
            notyf()->error('Doctor Staff Authentication Failed User Is Not A Staff.');
            return false;
        }
        return true;
    }

    protected function getDeviceName($userAgent): string
    {
        if (strpos($userAgent, 'Windows') !== false)
            return 'Windows Device';
        if (strpos($userAgent, 'Mac') !== false)
            return 'Mac Device';
        if (strpos($userAgent, 'Linux') !== false)
            return 'Linux Device';
        if (strpos($userAgent, 'iPhone') !== false)
            return 'iPhone';
        if (strpos($userAgent, 'Android') !== false)
            return 'Android Device';
        return 'Unknown Device';
    }

    protected function getDashboardRoute(): string
    {
        return match ($this->role) {
            'doctor' => route('doctor.doctor-dashboard'),
            'doctor_staff' => route('doctor.staff.doctor-dashboard'),
            default => route('home')
        };
    }

    public function render()
    {
        return view('livewire.doctor.auth.login-form', [
            'isMobileLogin' => $this->useMobile
        ]);
    }
}
