<?php

namespace App\Livewire\Admin\SubscriptionPlanWizard;

use App\Models\SubscriptionPlan;
use App\Models\AddOn;
use Livewire\Component;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Log;

class Step1PlanDetails extends Component
{
    public $plan_code;
    public $plan_name;
    public $description;
    public $plan_type = 'monthly';
    public $duration = '1';
    public $base_price;
    public $discounted_price;
    public $addon_eligible = true;
    public $auto_renewal = true;
    public $is_combo = false;
    public $plan_status = 'active';
    public $currency = 'USD';
    public ?int $editingPlanId = null;
    public $max_doctors = 1;
    public $max_staff = 1;
    public $device_limit = 1;
    public $selected_plan_id;
    public $availablePlans;
    public $selectedPlanDetails;
    public $selectedPlanLimits;
    public $selected_add_on_id = [];
    public $availableAddOns;
    public $selectedAddOnDetails = [];
    public $comboItems = [];
    public $comboPlanPrice = 0;
    public $comboTotalPrice = 0;

    public function mount($data = [], $editingPlanId = null)
    {
        Log::debug('Step1PlanDetails mount called', [
            'editingPlanId' => $editingPlanId,
            'incoming_data' => $data
        ]);

        $this->editingPlanId = $editingPlanId;
        $this->availablePlans = SubscriptionPlan::where('plan_status', 'active')->where('is_combo' , false)->get();
        $this->availableAddOns = AddOn::where('is_active', true)->get();

        if (!empty($data['basic'])) {
            $data['basic']['addon_eligible'] = (bool) ($data['basic']['addon_eligible'] ?? true);
            $data['basic']['auto_renewal'] = (bool) ($data['basic']['auto_renewal'] ?? true);
            $data['basic']['is_combo'] = (bool) ($data['basic']['is_combo'] ?? false);
            Log::debug('Filling component with existing data', $data);
            $this->fill($data['basic']);
        }

        if (!empty($data['combo'])) {
            $this->selected_add_on_id = $data['combo']['addon'];
            $this->selected_plan_id = $data['combo']['plan'];
            $this->selectedPlanDetails = SubscriptionPlan::find($this->selected_plan_id);
        }

        Log::debug('Current component state after mount', $this->getState());
    }

    public function rules()
    {
        $rules = [
            'plan_code' => [
                'required',
                'string',
                'max:50',
                Rule::unique('subscription_plans', 'plan_code')->ignore($this->editingPlanId),
            ],
            'plan_name' => 'required|string|max:100',
            'description' => 'nullable|string|max:500',
            'plan_type' => 'required|in:monthly,annual,custom',
            'duration' => 'required|integer|min:1',
            'base_price' => 'required|numeric|min:0',
            'discounted_price' => 'nullable|numeric|min:0|lt:base_price',
            'addon_eligible' => 'boolean',
            'is_combo' => 'boolean',
            'plan_status' => 'required|in:active,inactive',
            'currency' => 'required|in:INR,USD,EUR,GBP',
            'max_doctors' => 'required|integer|min:1',
            'max_staff' => 'required|integer|min:1',
            'device_limit' => 'required|integer|min:1',
        ];

        if ($this->is_combo) {
            $rules['selected_plan_id'] = [
                'nullable',
                'integer',
                'min:1',
                Rule::exists('subscription_plans', 'id')->where('plan_status', 'active'),
                'required_without_all:selected_add_on_id',
            ];

            $rules['selected_add_on_id'] = [
                'nullable',
                'array',
                'required_without_all:selected_plan_id',
                function ($attribute, $value, $fail) {
                    if (count($value) > 0) {
                        $invalidAddons = array_diff($value, $this->availableAddOns->pluck('id')->toArray());
                        if (!empty($invalidAddons)) {
                            $fail('One or more selected add-ons are invalid.');
                        }
                    }
                }
            ];
        }

        Log::debug('Validation rules being applied', [
            'rules' => $rules,
            'current_plan_code' => $this->plan_code,
            'editingPlanId' => $this->editingPlanId
        ]);

        return $rules;
    }

    public function messages()
    {
        return [
            'selected_plan_id.required_without_all' => 'For a combo, please select a plan or at least one add-on.',
            'selected_add_on_id.required_without_all' => 'For a combo, please select a plan or at least one add-on.',
        ];
    }

    public function validationAttributes()
    {
        return [
            'plan_code' => 'Plan Code',
            'plan_name' => 'Plan Name',
            'plan_type' => 'Plan Type',
            'base_price' => 'Base Price',
            'discounted_price' => 'Discounted Price',
            'selected_plan_id' => 'Combo Subscription Plan',
            'selected_add_on_id' => 'Combo Add On',
        ];
    }

    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);

        if ($propertyName === 'base_price' && $this->discounted_price !== null) {
            $this->validateOnly('discounted_price');
        }

        if ($propertyName === 'is_combo' || str_starts_with($propertyName, 'selected_add_on_id')) {
            $this->validateOnly('selected_plan_id');
            $this->validateOnly('selected_add_on_id');
        }

        if ($propertyName === 'selected_plan_id') {
            $this->validateOnly('selected_add_on_id');
        }
    }

    public function next()
    {
        Log::debug('Attempting to validate step 1 data', $this->getState());

        $validatedData = $this->validate();
        Log::info('Step 1 validation passed', $validatedData);

        // Create a filtered version of the data without combo fields
        $basicData = $this->filterBasicData($validatedData);

        // Dispatch basic data (without combo fields)
        $this->dispatch('updatePlanData', step: 'basic', data: $basicData);

        // If it's a combo plan, dispatch combo data separately
        if ($this->is_combo) {
            $comboData = [
                'plan' => $this->selected_plan_id,
                'addon' => $this->selected_add_on_id,
            ];
            $this->dispatch('updatePlanData', step: 'is_combo', data: $comboData);
        }

        $this->dispatch('goToStep', step: 2);
    }

    protected function filterBasicData(array $validatedData): array
    {
        // Fields to exclude from basic data
        $excludeFields = [
            'selected_plan_id',
            'selected_add_on_id',
            // Add any other combo-specific fields here
        ];

        return array_diff_key($validatedData, array_flip($excludeFields));
    }

    protected function getState()
    {
        return $this->only([
            'plan_code',
            'plan_name',
            'description',
            'plan_type',
            'duration',
            'base_price',
            'discounted_price',
            'addon_eligible',
            'is_combo',
            'plan_status',
            'currency',
            'editingPlanId',
            'auto_renewal',
        ]);
    }

    public function updateSelectedCount()
    {
        // Just forces a UI refresh
        $this->dispatch('addons-updated');
        if ($this->selected_add_on_id) {
            foreach ($this->selected_add_on_id as $index => $addOnId) {
                $this->selectedAddOnDetails[$index] = AddOn::find($addOnId);
            }
        } else {
            $this->selectedAddOnDetails = null;
        }
    }

    public function updatedSelectedPlanId($value)
    {
        if ($value) {
            $this->selectedPlanDetails = SubscriptionPlan::find($value);
            $this->selectedPlanLimits = $this->selectedPlanDetails->limit;
        } else {
            $this->selectedPlanDetails = null;
        }
    }

    public function updatedIsCombo($value)
    {
        if (!$value) {
            $this->reset(['selected_plan_id', 'selected_add_on_id', 'selectedPlanDetails', 'selectedAddOnDetails']);
        }
    }

    public function render()
    {
        return view('livewire.admin.subscription-plan-wizard.step1-plan-details');
    }
}
