<?php

namespace App\Livewire\Admin\Plans;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\SubscriptionPlan;
use Illuminate\Support\Facades\Log;

class SubscriptionPlansList extends Component
{
    use WithPagination;

    public $search = '';
    public $statusFilter = 'all';
    public $typeFilter = 'all';
    public $perPage = 10;
    public $sortField = 'plan_name';
    public $sortDirection = 'asc';
    public $selectedPlanId = null;
    public $showDeleteModal = false;
    public $deleteId = null;

    protected $queryString = [
        'search' => ['except' => ''],
        'statusFilter' => ['except' => 'all'],
        'typeFilter' => ['except' => 'all'],
        'perPage' => ['except' => 10],
        'sortField' => ['except' => 'plan_name'],
        'sortDirection' => ['except' => 'asc'],
    ];

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function applyFilters()
    {
        $this->resetPage();
    }

    public function clearAllFilters()
    {
        $this->statusFilter = 'all';
        $this->typeFilter = 'all';
        $this->perPage = 10;
        $this->search = '';
        $this->resetPage();
    }

    public function resetStatusFilter()
    {
        $this->statusFilter = 'all';
        $this->resetPage();
    }

    public function resetTypeFilter()
    {
        $this->typeFilter = 'all';
        $this->resetPage();
    }

    public function resetPerPageFilter()
    {
        $this->perPage = 10;
        $this->resetPage();
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDirection = 'asc';
        }
        $this->resetPage();
    }

    public function confirmDelete($planId)
    {
        $this->selectedPlanId = $planId;
        $this->dispatch('showDeleteModal');
    }

    public function toggleStatus($id)
    {
        $plan = SubscriptionPlan::findOrFail($id);
        $newStatus = $plan->plan_status === 'active' ? 'inactive' : 'active';
        $plan->update(['plan_status' => $newStatus]);

        notyf()->success('Plan status updated successfully');
    }

    public function delete($id)
    {
        try {
            $plan = SubscriptionPlan::findOrFail($id);
            $plan->delete();

            $this->dispatch('hideDeleteModal');
            notyf()->success('Plan deleted successfully!');
        } catch (\Exception $e) {
            Log::error("Error deleting plan: " . $e->getMessage());
            notyf()->error('Failed to delete plan.');
        }

        $this->reset(['selectedPlanId', 'showDeleteModal']);
    }

    public function render()
    {
        $plans = SubscriptionPlan::query()
            ->with(['limit', 'features.feature', 'usageRates.usageType'])
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('plan_name', 'like', '%' . $this->search . '%')
                        ->orWhere('plan_code', 'like', '%' . $this->search . '%')
                        ->orWhere('description', 'like', '%' . $this->search . '%');
                });
            })
            ->when($this->statusFilter !== 'all', function ($query) {
                $query->where('plan_status', $this->statusFilter);
            })
            ->when($this->typeFilter !== 'all', function ($query) {
                $query->where('plan_type', $this->typeFilter);
            })
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->perPage);

        return view('livewire.admin.plans.subscription-plans-list', [
            'plans' => $plans,
            'statusOptions' => [
                'all' => 'All Statuses',
                'active' => 'Active',
                'inactive' => 'Inactive',
                'draft' => 'Draft',
            ],
            'typeOptions' => [
                'all' => 'All Types',
                'monthly' => 'Monthly',
                'annual' => 'Annual',
                'custom' => 'Custom',
            ],
        ]);
    }
}
