<?php

namespace App\Livewire\Admin\Plans;

use Livewire\Component;

use App\Models\Feature;
use App\Models\UsageType;
use Illuminate\Support\Facades\Log;

class SubscriptionPlanView extends Component
{
    public $planData;
    public $plan;
    public $available_features;
    public $available_usage_types;

    public function mount($plan)
    {
        $this->plan = $plan;
        $this->loadPlanData();
        $this->validateData();

        $this->available_features = Feature::whereIn('id', $this->planData['features'] ?? [])->get();
        $this->available_usage_types = UsageType::all()->keyBy('id');
    }

    public function loadPlanData()
    {
        try {

            $plan = $this->plan;
            $basicFields = [
                'plan_code',
                'plan_name',
                'description',
                'plan_type',
                'duration',
                'base_price',
                'discounted_price',
                'addon_eligible',
                'is_combo',
                'plan_status',
                'max_doctors',
                'max_staff',
                'device_limit',
                'trial_period',
                'auto_renewal',
                'currency'
            ];

            $this->planData['basic'] = $plan->only($basicFields);

            if ($plan->limit) {
                $limitFields = [
                    'included_doctors',
                    'included_staff',
                    'storage_limit_gb',
                    'teleconsultation_minutes',
                    'sms_limit',
                    'whatsapp_limit',
                    'additional_doctor_price',
                    'additional_staff_price'
                ];
                $this->planData['limits'] = $plan->limit->only($limitFields);
                Log::info("Stored Limit Data:", $this->planData['limits']);
            } else {
                $this->planData['limits'] = [];
            }

            $this->planData['features'] = $plan->features->pluck('feature_id')->toArray();
            
            $this->planData['rates'] = $plan->usageRates->map(function ($rate) {
                return [
                    'usage_type_id' => $rate->usage_type_id,
                    'rate_per_unit' => $rate->rate_per_unit,
                    'free_units' => $rate->free_units,
                    'overage_rate' => $rate->overage_rate,
                    'usage_type_name' => $rate->usageType->name ?? 'Unknown'
                ];
            })->toArray();
        } catch (\Exception $e) {
            Log::error("Error loading plan data: " . $e->getMessage());
            notyf()->error('Failed to load plan data');
        }
    }

    protected function validateData()
    {
        if (
            empty($this->planData['basic']) ||
            empty($this->planData['limits']) ||
            empty($this->planData['features'])
        ) {
            notyf()->error( 'Missing required plan data');
            $this->dispatch('goToStep', step: 1);
        }
    }

    public function render()
    {
        return view('livewire.admin.plans.subscription-plan-view');
    }
}