<?php

namespace App\Livewire\Admin\Plans;

use Livewire\Component;
use App\Models\SubscriptionPlan;
use App\Models\PlanLimit;
use App\Models\Feature;
use App\Models\PlanFeature;
use App\Models\UsageType;
use App\Models\UsageRate;
use App\Models\AddOn;
use Livewire\WithPagination;

class SubscriptionPlanManagement extends Component
{
    use WithPagination;

    public $planId;
    public $plan_code;
    public $plan_name;
    public $description;
    public $plan_type = 'monthly';
    public $duration = '1';
    public $base_price;
    public $discounted_price;
    public $addon_eligible = true;
    public $is_combo = false;
    public $plan_status = 'active';
    public $max_doctors;
    public $max_staff;
    public $device_limit;
    public $trial_period;
    public $auto_renewal = true;
    public $currency = 'INR';

    public $included_doctors;
    public $included_staff;
    public $storage_limit_gb;
    public $teleconsultation_minutes;
    public $sms_limit;
    public $whatsapp_limit;
    public $additional_doctor_price;
    public $additional_staff_price;

    public $usage_rates = [];
    public $available_usage_types = [];
    public $current_usage_type;
    public $current_rate_per_unit = 0;
    public $current_free_units = 0;
    public $current_overage_rate = 0;

    public $selected_features = [];
    public $available_features = [];

    public $addons = [];
    public $available_addons = [];

    public $showDeleteModal = false;
    public $editingPlanId = null;
    public $activeTab = 'basic';

    public $viewingPlan = null;
    public $showViewModal = false;

    protected $rules = [
        'plan_code' => 'required|unique:subscription_plans,plan_code',
        'plan_name' => 'required',
        'description' => 'nullable',
        'plan_type' => 'required|in:monthly,annual,custom',
        'duration' => 'required',
        'base_price' => 'required|numeric|min:0',
        'discounted_price' => 'nullable|numeric|min:0',
        'addon_eligible' => 'boolean',
        'is_combo' => 'boolean',
        'plan_status' => 'required|in:active,inactive',
        'max_doctors' => 'required|integer|min:1',
        'max_staff' => 'required|integer|min:1',
        'device_limit' => 'required|integer|min:1',
        'trial_period' => 'required|integer|min:0',
        'auto_renewal' => 'boolean',
        'currency' => 'required|in:INR,USD,EUR,GBP',

        'included_doctors' => 'required|integer|min:0',
        'included_staff' => 'required|integer|min:0',
        'storage_limit_gb' => 'required|integer|min:0',
        'teleconsultation_minutes' => 'required|integer|min:0',
        'sms_limit' => 'required|integer|min:0',
        'whatsapp_limit' => 'required|integer|min:0',
        'additional_doctor_price' => 'required|numeric|min:0',
        'additional_staff_price' => 'required|numeric|min:0',
    ];

    public function mount()
    {
        $this->available_usage_types = UsageType::where('is_active', true)->get();
        $this->available_features = Feature::where('feature_status', 'active')->get();
        $this->available_addons = AddOn::where('is_active', true)->get();
    }

    public function render()
    {
        return view('livewire.admin.plans.subscription-plan-management', [
            'plans' => SubscriptionPlan::with(['limit', 'features.feature', 'usageRates.usageType'])
                ->paginate(10)
        ]);
    }

    public function viewPlan($id)
    {
        $this->viewingPlan = SubscriptionPlan::with(['limit', 'features.feature', 'usageRates.usageType'])
            ->findOrFail($id);
        $this->showViewModal = true;
    }

    public function closeViewModal()
    {
        $this->showViewModal = false;
        $this->viewingPlan = null;
    }

    public function createPlan()
    {
        $this->validate();

        $plan = SubscriptionPlan::create([
            'plan_code' => $this->plan_code,
            'plan_name' => $this->plan_name,
            'description' => $this->description,
            'plan_type' => $this->plan_type,
            'duration' => $this->duration,
            'base_price' => $this->base_price,
            'discounted_price' => $this->discounted_price,
            'addon_eligible' => $this->addon_eligible,
            'is_combo' => $this->is_combo,
            'plan_status' => $this->plan_status,
            'max_doctors' => $this->max_doctors,
            'max_staff' => $this->max_staff,
            'device_limit' => $this->device_limit,
            'trial_period' => $this->trial_period,
            'auto_renewal' => $this->auto_renewal,
            'currency' => $this->currency,
        ]);

        PlanLimit::create([
            'plan_id' => $plan->id,
            'included_doctors' => $this->included_doctors,
            'included_staff' => $this->included_staff,
            'storage_limit_gb' => $this->storage_limit_gb,
            'teleconsultation_minutes' => $this->teleconsultation_minutes,
            'sms_limit' => $this->sms_limit,
            'whatsapp_limit' => $this->whatsapp_limit,
            'additional_doctor_price' => $this->additional_doctor_price,
            'additional_staff_price' => $this->additional_staff_price,
        ]);

        foreach ($this->selected_features as $featureId) {
            PlanFeature::create([
                'plan_id' => $plan->id,
                'feature_id' => $featureId,
                'included' => true
            ]);
        }

        foreach ($this->usage_rates as $usageRate) {
            foreach ($usageRate['usage_type_ids'] as $usageTypeId) {
                UsageRate::create([
                    'plan_id' => $plan->id,
                    'usage_type_id' => $usageTypeId,
                    'rate_per_unit' => $usageRate['rate_per_unit'],
                    'free_units' => $usageRate['free_units'],
                    'overage_rate' => $usageRate['overage_rate'],
                    'currency' => $this->currency,
                    'effective_date' => now(),
                ]);
            }
        }

        notyf()->success('Plan created successfully!');
        $this->resetForm();
    }

    public function editPlan($id)
    {
        $plan = SubscriptionPlan::with(['limit', 'features', 'usageRates.usageType'])->findOrFail($id);

        $this->editingPlanId = $id;
        $this->plan_code = $plan->plan_code;
        $this->plan_name = $plan->plan_name;
        $this->description = $plan->description;
        $this->plan_type = $plan->plan_type;
        $this->duration = $plan->duration;
        $this->base_price = $plan->base_price;
        $this->discounted_price = $plan->discounted_price;
        $this->addon_eligible = $plan->addon_eligible;
        $this->is_combo = $plan->is_combo;
        $this->plan_status = $plan->plan_status;
        $this->max_doctors = $plan->max_doctors;
        $this->max_staff = $plan->max_staff;
        $this->device_limit = $plan->device_limit;
        $this->trial_period = $plan->trial_period;
        $this->auto_renewal = $plan->auto_renewal;
        $this->currency = $plan->currency;

        if ($plan->limit) {
            $this->included_doctors = $plan->limit->included_doctors;
            $this->included_staff = $plan->limit->included_staff;
            $this->storage_limit_gb = $plan->limit->storage_limit_gb;
            $this->teleconsultation_minutes = $plan->limit->teleconsultation_minutes;
            $this->sms_limit = $plan->limit->sms_limit;
            $this->whatsapp_limit = $plan->limit->whatsapp_limit;
            $this->additional_doctor_price = $plan->limit->additional_doctor_price;
            $this->additional_staff_price = $plan->limit->additional_staff_price;
        }

        $this->selected_features = $plan->features->pluck('feature_id')->toArray();

        $this->usage_rates = $plan->usageRates->map(function ($rate) {
            return [
                'id' => $rate->id,
                'usage_type_ids' => [$rate->usage_type_id],
                'rate_per_unit' => $rate->rate_per_unit,
                'free_units' => $rate->free_units,
                'overage_rate' => $rate->overage_rate,
                'currency' => $rate->currency,
                'effective_date' => $rate->effective_date,
            ];
        })->toArray();
    }

    public function updatePlan()
    {
        $this->validate([
            'plan_code' => 'required|unique:subscription_plans,plan_code,' . $this->editingPlanId,
            'plan_name' => 'required',
            'description' => 'nullable',
            'plan_type' => 'required|in:monthly,annual,custom',
            'duration' => 'required',
            'base_price' => 'required|numeric|min:0',
            'discounted_price' => 'nullable|numeric|min:0',
            'addon_eligible' => 'boolean',
            'is_combo' => 'boolean',
            'plan_status' => 'required|in:active,inactive',
            'max_doctors' => 'required|integer|min:1',
            'max_staff' => 'required|integer|min:1',
            'device_limit' => 'required|integer|min:1',
            'trial_period' => 'required|integer|min:0',
            'auto_renewal' => 'boolean',
            'currency' => 'required|string|size:3',

            'included_doctors' => 'required|integer|min:0',
            'included_staff' => 'required|integer|min:0',
            'storage_limit_gb' => 'required|integer|min:0',
            'teleconsultation_minutes' => 'required|integer|min:0',
            'sms_limit' => 'required|integer|min:0',
            'whatsapp_limit' => 'required|integer|min:0',
            'additional_doctor_price' => 'required|numeric|min:0',
            'additional_staff_price' => 'required|numeric|min:0',
        ]);

        $plan = SubscriptionPlan::findOrFail($this->editingPlanId);
        $plan->update([
            'plan_code' => $this->plan_code,
            'plan_name' => $this->plan_name,
            'description' => $this->description,
            'plan_type' => $this->plan_type,
            'duration' => $this->duration,
            'base_price' => $this->base_price,
            'discounted_price' => $this->discounted_price,
            'addon_eligible' => $this->addon_eligible,
            'is_combo' => $this->is_combo,
            'plan_status' => $this->plan_status,
            'max_doctors' => $this->max_doctors,  // Changed from max_users
            'max_staff' => $this->max_staff,      // New field
            'device_limit' => $this->device_limit,
            'trial_period' => $this->trial_period,
            'auto_renewal' => $this->auto_renewal,
            'currency' => $this->currency,
        ]);

        $plan->limit()->updateOrCreate(
            ['plan_id' => $plan->id],
            [
                'included_doctors' => $this->included_doctors,
                'included_staff' => $this->included_staff,
                'storage_limit_gb' => $this->storage_limit_gb,
                'teleconsultation_minutes' => $this->teleconsultation_minutes,
                'sms_limit' => $this->sms_limit,
                'whatsapp_limit' => $this->whatsapp_limit,
                'additional_doctor_price' => $this->additional_doctor_price,
                'additional_staff_price' => $this->additional_staff_price,
            ]
        );

        $plan->features()->delete();
        foreach ($this->selected_features as $featureId) {
            PlanFeature::create([
                'plan_id' => $plan->id,
                'feature_id' => $featureId,
                'included' => true
            ]);
        }

        foreach ($this->usage_rates as $usageRate) {
            if (isset($usageRate['id'])) {
                UsageRate::where('id', $usageRate['id'])->update([
                    'usage_type_id' => $usageRate['usage_type_ids'][0],
                    'rate_per_unit' => $usageRate['rate_per_unit'],
                    'free_units' => $usageRate['free_units'],
                    'overage_rate' => $usageRate['overage_rate'],
                    'currency' => $this->currency,
                    'effective_date' => now(),
                ]);

                if (count($usageRate['usage_type_ids']) > 1) {
                    for ($i = 1; $i < count($usageRate['usage_type_ids']); $i++) {
                        UsageRate::create([
                            'plan_id' => $plan->id,
                            'usage_type_id' => $usageRate['usage_type_ids'][$i],
                            'rate_per_unit' => $usageRate['rate_per_unit'],
                            'free_units' => $usageRate['free_units'],
                            'overage_rate' => $usageRate['overage_rate'],
                            'currency' => $this->currency,
                            'effective_date' => now(),
                        ]);
                    }
                }
            } else {
                foreach ($usageRate['usage_type_ids'] as $usageTypeId) {
                    UsageRate::create([
                        'plan_id' => $plan->id,
                        'usage_type_id' => $usageTypeId,
                        'rate_per_unit' => $usageRate['rate_per_unit'],
                        'free_units' => $usageRate['free_units'],
                        'overage_rate' => $usageRate['overage_rate'],
                        'currency' => $this->currency,
                        'effective_date' => now(),
                    ]);
                }
            }
        }

        notyf()->success('Plan updated successfully!');
        $this->resetForm();
    }

    public function confirmDelete($id)
    {
        $this->planId = $id;
        $this->showDeleteModal = true;
    }

    public function deletePlan()
    {
        $plan = SubscriptionPlan::findOrFail($this->planId);
        $plan->delete();

        notyf()->success('Plan deleted successfully!');
        $this->showDeleteModal = false;
    }

    public function closeModal()
    {
        $this->showDeleteModal = false;
    }

    public function resetForm()
    {
        $this->reset();
        $this->mount();
    }

    public function addUsageRate()
    {
        $this->validate([
            'current_usage_type' => 'required|exists:usage_types,id',
            'current_rate_per_unit' => 'required|numeric|min:0',
            'current_free_units' => 'required|integer|min:0',
            'current_overage_rate' => 'required|numeric|min:0',
        ]);
    
        $existingIndex = null;
        foreach ($this->usage_rates as $index => $rate) {
            if (in_array($this->current_usage_type, $rate['usage_type_ids'])) {
                $existingIndex = $index;
                break;
            }
        }
    
        $newRate = [
            'usage_type_ids' => [$this->current_usage_type],
            'rate_per_unit' => $this->current_rate_per_unit,
            'free_units' => $this->current_free_units,
            'overage_rate' => $this->current_overage_rate,
        ];
    
        if ($existingIndex !== null) {
            $this->usage_rates[$existingIndex] = $newRate;
        } else {
            $this->usage_rates[] = $newRate;
        }
    
        $this->resetCurrentUsageRate();
    }
    
    protected function resetCurrentUsageRate()
    {
        $this->reset(['current_usage_type', 'current_rate_per_unit', 'current_free_units', 'current_overage_rate']);
    }

    public function removeUsageRate($index)
    {
        if ($this->editingPlanId && isset($this->usage_rates[$index]['id'])) {
            UsageRate::where('id', $this->usage_rates[$index]['id'])->delete();
        }
        
        unset($this->usage_rates[$index]);
        $this->usage_rates = array_values($this->usage_rates);
        
        notyf()->success('Usage rate removed successfully!');
    }
}
