<?php

namespace App\Livewire\Admin\Plans;

use Livewire\Component;
use App\Models\Feature;
use Illuminate\Support\Str;
use Livewire\WithPagination;

class FeaturesList extends Component
{
    use WithPagination;

    public $search = '';
    public $perPage = 10;
    public $sortField = 'feature_name';
    public $sortDirection = 'asc';
    public $statusFilter = 'all';
    public $baseFeatureFilter = 'all';
    public $categoryFilter = 'all';
    public $deleteId = null;

    protected $queryString = [
        'search' => ['except' => ''],
        'sortField' => ['except' => 'feature_name'],
        'sortDirection' => ['except' => 'asc'],
        'statusFilter' => ['except' => 'all'],
        'baseFeatureFilter' => ['except' => 'all'],
        'categoryFilter' => ['except' => 'all']
    ];

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function applyFilters()
    {
        $this->resetPage();
    }

    public function clearAllFilters()
    {
        $this->statusFilter = 'all';
        $this->baseFeatureFilter = 'all';
        $this->categoryFilter = 'all';
        $this->search = '';
        $this->resetPage();
    }

    public function resetStatusFilter()
    {
        $this->statusFilter = 'all';
        $this->resetPage();
    }

    public function resetBaseFeatureFilter()
    {
        $this->baseFeatureFilter = 'all';
        $this->resetPage();
    }

    public function resetCategoryFilter()
    {
        $this->categoryFilter = 'all';
        $this->resetPage();
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
        $this->resetPage();
    }

    public function toggleStatus($id)
    {
        $feature = Feature::findOrFail($id);
        $newStatus = $feature->feature_status === 'active' ? 'inactive' : 'active';
        $feature->update(['feature_status' => $newStatus]);

        notyf()->success('Feature status updated successfully');
    }

    public function delete($id)
    {
        try {
            $feature = Feature::findOrFail($id);
            $featureName = $feature->feature_name;

            if ($feature->planFeatures()->exists()) {
                $planCount = $feature->planFeatures()->count();
                throw new \Exception("This feature is currently used in {$planCount} plan(s). Please remove it from all plans before deleting.");
            }

            $feature->delete();
            notyf()->success("Feature '{$featureName}' deleted successfully");
        } catch (\Exception $e) {
            notyf()->error("Cannot delete feature: " . $e->getMessage());
        }

        $this->deleteId = null;
    }

    public function render()
    {
        $features = Feature::query()
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('feature_name', 'like', '%' . $this->search . '%')
                        ->orWhere('feature_code', 'like', '%' . $this->search . '%')
                        ->orWhere('description', 'like', '%' . $this->search . '%')
                        ->orWhere('category', 'like', '%' . $this->search . '%');
                });
            })
            ->when($this->statusFilter !== 'all', function ($query) {
                $query->where('feature_status', $this->statusFilter);
            })
            ->when($this->baseFeatureFilter !== 'all', function ($query) {
                if ($this->baseFeatureFilter === 'yes') {
                    $query->where('is_base_feature', true);
                } else {
                    $query->where('is_base_feature', false);
                }
            })
            ->when($this->categoryFilter !== 'all' && !empty($this->categoryFilter), function ($query) {
                $query->where('category', $this->categoryFilter);
            })
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->perPage);

        $categories = Feature::select('category')
            ->whereNotNull('category')
            ->where('category', '!=', '')
            ->distinct()
            ->orderBy('category')
            ->pluck('category');

        return view('livewire.admin.plans.features-list', [
            'features' => $features,
            'categories' => $categories
        ]);
    }
}
