<?php

namespace App\Livewire\Admin\Doctor;

use App\Models\User;
use App\Models\Doctor;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Validation\Rule;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Hash;

class DoctorManagement extends Component
{
    use WithPagination;

    public $name, $email, $password, $password_confirmation;
    public $specialization, $qualifications, $experience_years, $license_number, $bio;
    public $status = 'active';
    public $editMode = false;
    public $userIdBeingEdited = null;
    public $searchTerm = '';

    protected $rules = [
        'name' => 'required|string|max:255',
        'email' => 'required|email|max:255|unique:users,email',
        'password' => 'required|min:6|confirmed',
        'specialization' => 'required|string|max:255',
        'qualifications' => 'nullable|string',
        'experience_years' => 'nullable|integer|min:0',
        'license_number' => 'required|string|max:255|unique:doctors,license_number',
        'bio' => 'nullable|string',
        'status' => 'required|in:active,inactive',
    ];

    public function mount()
    {
        Gate::authorize('list doctors');
    }

    public function render()
    {
        $searchTerm = '%' . $this->searchTerm . '%';

        $users = User::where(function ($q) use ($searchTerm) {
            $q->where('name', 'like', $searchTerm)
                ->orWhere('email', 'like', $searchTerm);
        })
            ->whereHas('roles', function ($q) {
                $q->where('group', 'doctor_group');
            })
            ->with(['doctor', 'roles'])
            ->orderBy('id', 'desc')
            ->paginate(10);

        return view('livewire.admin.doctor.doctor-management', [
            'users' => $users,
        ]);
    }

    public function resetForm()
    {
        $this->reset([
            'name',
            'email',
            'password',
            'password_confirmation',
            'specialization',
            'qualifications',
            'experience_years',
            'license_number',
            'bio',
            'status',
            'userIdBeingEdited',
            'editMode'
        ]);

        $this->resetValidation();
    }

    public function save()
    {
        if ($this->editMode) {
            $this->updateDoctor();
        } else {
            $this->createDoctor();
        }
    }

    public function createDoctor()
    {
        Gate::authorize('create doctors');
        $this->validate();

        try {
            $user = User::create([
                'name' => $this->name,
                'email' => $this->email,
                'password' => Hash::make($this->password),
                'status' => $this->status,
            ]);

            $doctorRole = Role::where('name', 'doctor')
                ->where('guard_name', 'web')
                ->firstOrFail();

            $user->assignRole($doctorRole);

            Doctor::create([
                'user_id' => $user->id,
                'specialization' => $this->specialization,
                'qualifications' => $this->qualifications,
                'experience_years' => $this->experience_years,
                'license_number' => $this->license_number,
                'bio' => $this->bio,
            ]);

            notyf()->success('Doctor created successfully.');
            $this->resetForm();
        } catch (\Exception $e) {
            Log::error('Doctor creation failed: ' . $e->getMessage());
            notyf()->error('Failed to create doctor: ' . $e->getMessage());
        }
    }
    public function edit($id)
    {
        Gate::authorize('edit doctors');
        $user = User::with(['doctor', 'roles'])->findOrFail($id);

        $this->editMode = true;
        $this->userIdBeingEdited = $id;
        $this->name = $user->name;
        $this->email = $user->email;
        $this->status = $user->status;

        if ($user->doctor) {
            $this->specialization = $user->doctor->specialization;
            $this->qualifications = $user->doctor->qualifications;
            $this->experience_years = $user->doctor->experience_years;
            $this->license_number = $user->doctor->license_number;
            $this->bio = $user->doctor->bio;
        }
    }

    public function updateDoctor()
    {
        Gate::authorize('edit doctors');
        $rules = [
            'name' => 'required|string|max:255',
            'email' => [
                'required',
                'email',
                'max:255',
                Rule::unique('users', 'email')->ignore($this->userIdBeingEdited)
            ],
            'specialization' => 'required|string|max:255',
            'qualifications' => 'nullable|string',
            'experience_years' => 'nullable|integer|min:0',
            'license_number' => [
                'required',
                'string',
                'max:255',
                Rule::unique('doctors', 'license_number')->ignore($this->userIdBeingEdited, 'user_id')
            ],
            'bio' => 'nullable|string',
            'status' => 'required|in:active,inactive',
        ];

        if ($this->password) {
            $rules['password'] = 'nullable|min:6|confirmed';
        }

        $this->validate($rules);

        $user = User::findOrFail($this->userIdBeingEdited);

        $user->update([
            'name' => $this->name,
            'email' => $this->email,
            'status' => $this->status,
            'password' => $this->password ? Hash::make($this->password) : $user->password,
        ]);

        Doctor::updateOrCreate(
            ['user_id' => $user->id],
            [
                'specialization' => $this->specialization,
                'qualifications' => $this->qualifications,
                'experience_years' => $this->experience_years,
                'license_number' => $this->license_number,
                'bio' => $this->bio,
            ]
        );

        notyf()->success('Doctor updated successfully.');
        $this->resetForm();
    }

    public function deleteDoctor($id)
    {   Gate::authorize('delete doctors');
        if (auth()->id() == $id) {
            notyf()->error('You cannot delete your own account.');
            return;
        }

        $user = User::findOrFail($id);

        if ($user->doctor) {
            $user->doctor->delete();
        }

        $user->delete();

        notyf()->success('Doctor deleted successfully.');
    }
}