<?php

namespace App\Livewire\Admin\Doctor;

use App\Models\User;
use App\Models\Clinic;
use App\Models\Doctor;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;

class DoctorList extends Component
{
    use WithPagination;

    protected $paginationTheme = 'bootstrap';

    public $searchTerm = '';
    public $statusFilter = 'all';
    public $specializationFilter = 'all';
    public $clinicFilter = 'all';
    public $perPage = 10;
    public $sortField = 'name';
    public $sortDirection = 'asc';
    public $sortBy = 'recent';

    protected $queryString = [
        'searchTerm' => ['except' => ''],
        'statusFilter' => ['except' => 'all'],
        'specializationFilter' => ['except' => 'all'],
        'clinicFilter' => ['except' => 'all'],
        'perPage' => ['except' => 10],
        'sortField' => ['except' => 'name'],
        'sortDirection' => ['except' => 'asc'],
    ];

    public function toggleStatus($id)
    {
        Gate::authorize('edit doctors');
        $user = User::findOrFail($id);
        $user->status = $user->status === 'active' ? 'inactive' : 'active';
        $user->save();

        notyf()->success('Doctor status updated successfully.');
    }

    public function deleteDoctor($id)
    {
        Gate::authorize('delete doctors');

        try {
            DB::transaction(function () use ($id) {
                $user = User::findOrFail($id);

                if ($user->doctor) {
                    $user->doctor->delete();
                }

                $user->clinics()->detach();

                $user->delete();
            });

            notyf()->success('Doctor deleted successfully.');
        } catch (\Exception $e) {
            notyf()->error('Failed to delete doctor: ' . $e->getMessage());
        }
    }
    public function resetFilters()
    {
        $this->reset([
            'statusFilter',
            'specializationFilter',
            'clinicFilter'
        ]);
    }

    public function applyFilters()
    {
        $this->resetPage();
    }

    public function getInitials($name)
    {
        if (empty($name)) {
            return 'NA';
        }

        $words = explode(' ', trim($name));

        if (count($words) >= 2) {
            return strtoupper(substr($words[0], 0, 1) . substr($words[count($words) - 1], 0, 1));
        } else {
            return strtoupper(substr($name, 0, 2));
        }
    }

    public function getAvatarColorClass($name)
    {
        $colors = ['bg-primary', 'bg-success', 'bg-info', 'bg-warning', 'bg-danger', 'bg-secondary'];
        $hash = crc32($name);
        return $colors[abs($hash) % count($colors)];
    }

    public function render()
    {
        $searchTerm = '%' . $this->searchTerm . '%';

        $query = User::where(function ($q) use ($searchTerm) {
            $q->where('name', 'like', $searchTerm)
                ->orWhere('email', 'like', $searchTerm)
                ->orWhere('status', 'like', $searchTerm)
                ->orWhereHas('doctor', function ($q) use ($searchTerm) {
                    $q->where('specialization', 'like', $searchTerm)
                        ->orWhere('license_number', 'like', $searchTerm);
                })
                ->orWhereHas('clinics', function ($q) use ($searchTerm) {
                    $q->where('name', 'like', $searchTerm);
                });
        })
            ->whereHas('roles', function ($q) {
                $q->where('group', 'doctor_group')
                    ->where('name', '!=', 'doctorstaff');
            })
            ->with([
                'doctor',
                'roles',
                'clinics' => function ($q) {
                    $q->withPivot('is_primary');
                }
            ]);

        // Rest of your existing filters (status, specialization, clinic, sorting)...
        if ($this->statusFilter !== 'all') {
            $query->where('status', $this->statusFilter);
        }

        if ($this->specializationFilter !== 'all') {
            $query->whereHas('doctor', function ($q) {
                $q->where('specialization', $this->specializationFilter);
            });
        }

        if ($this->clinicFilter !== 'all') {
            $query->whereHas('clinics', function ($q) {
                $q->where('clinics.id', $this->clinicFilter);
            });
        }

        // Sorting logic remains the same
        $query->when($this->sortBy === 'recent', function ($q) {
            $q->orderBy('created_at', 'desc');
        })
            ->when($this->sortBy === 'oldest', function ($q) {
                $q->orderBy('created_at', 'asc');
            })
            ->when(!in_array($this->sortBy, ['recent', 'oldest']), function ($q) {
                $q->orderBy($this->sortField, $this->sortDirection);
            });

        $totalDoctors = $query->count();
        $users = $query->paginate($this->perPage);

        $specializations = Doctor::select('specialization')
            ->distinct()
            ->whereNotNull('specialization')
            ->pluck('specialization')
            ->toArray();

        $clinics = Clinic::select('id', 'name')
            ->where('status', 'active')
            ->orderBy('name')
            ->get();

        return view('livewire.admin.doctor.doctor-list', [
            'users' => $users,
            'specializations' => $specializations,
            'clinics' => $clinics,
            'totalDoctors' => $totalDoctors,
        ]);
    }
}
