<?php

namespace App\Livewire\Admin\Doctor;

use App\Models\User;
use App\Models\Clinic;
use App\Models\Doctor;
use Livewire\Component;
use App\Models\DoctorClinic;
use App\Mail\Doctor\CreateDoctorMail;
use App\Mail\Doctor\UpdateDoctorMail;
use Illuminate\Support\Facades\Mail;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Livewire\Attributes\Computed;
use Livewire\WithFileUploads;

class DoctorForm extends Component
{
    use WithFileUploads;

    public $name, $email, $password, $password_confirmation;
    public $specialization, $qualifications, $experience_years, $license_number, $bio;
    public $status = 'active';
    public $editId = null;
    public $selected_clinics = [];
    public $primary_clinic_id = null;
    public $isEmbedded = false;
    public $clinicForm = false;
    public $has_primary_clinic = false;
    public $clinicSearch = '';
    public $showAllClinics = true;
    
    // Profile image properties
    public $profile_image;
    public $existing_profile_image = null;

    protected $rules = [
        'name' => 'required|string|max:255',
        'email' => 'required|email|max:255|unique:users,email',
        'password' => 'required|min:6|confirmed',
        'specialization' => 'required|string|max:255',
        'qualifications' => 'nullable|string',
        'experience_years' => 'nullable|integer|min:0',
        'license_number' => 'required|string|max:255|unique:doctors,license_number',
        'bio' => 'nullable|string',
        'status' => 'required|in:active,inactive',
        'selected_clinics' => 'nullable|array',
        'selected_clinics.*' => 'exists:clinics,id',
        'profile_image' => 'nullable|image|max:2048', // max 2MB
    ];

    public function mount($editId = null, $isEmbedded = false, $clinicForm = false)
    {
        $this->authorizeUser($editId);
        $this->initializeProperties($editId, $isEmbedded, $clinicForm);
        
        if ($editId) {
            $this->loadDoctorData($editId);
        } else {
            Log::info('Creating new doctor form initialized.');
        }
    }

    protected function authorizeUser($editId)
    {
        if ($editId) {
            Gate::authorize('edit doctors');
        } else {
            Gate::authorize('create doctors');
        }
    }

    protected function initializeProperties($editId, $isEmbedded, $clinicForm)
    {
        $this->isEmbedded = $isEmbedded;
        $this->clinicForm = $clinicForm;
        $this->editId = $editId;
    }

    protected function loadDoctorData($editId)
    {
        try {
            $user = User::with(['doctor', 'roles', 'clinics'])->findOrFail($editId);

            Log::info('Editing doctor form initialized.', ['user_id' => $editId]);

            $this->fillUserData($user);
            $this->fillDoctorData($user);
            $this->fillClinicData($user);

        } catch (\Exception $e) {
            Log::error('Failed to load doctor for editing: ' . $e->getMessage(), ['user_id' => $editId]);
            notyf()->error('Failed to load doctor details.');
        }
    }

    protected function fillUserData(User $user)
    {
        $this->name = $user->name;
        $this->email = $user->email;
        $this->status = $user->status;
    }

    protected function fillDoctorData(User $user)
    {
        if ($user->doctor) {
            $this->specialization = $user->doctor->specialization;
            $this->qualifications = $user->doctor->qualifications;
            $this->experience_years = $user->doctor->experience_years;
            $this->license_number = $user->doctor->license_number;
            $this->bio = $user->doctor->bio;
            $this->existing_profile_image = $user->doctor->profile_image;
            
            // Log for debugging
            Log::info('Doctor profile image loaded', [
                'doctor_id' => $user->id,
                'profile_image' => $this->existing_profile_image
            ]);
        }
    }

    protected function fillClinicData(User $user)
    {
        if ($user->clinics->isNotEmpty()) {
            $this->selected_clinics = $user->clinics->pluck('id')->toArray();

            $primaryClinic = $user->clinics->firstWhere('pivot.is_primary', true);
            if ($primaryClinic) {
                $this->primary_clinic_id = $primaryClinic->id;
                $this->has_primary_clinic = true;
            }           
        }
    }

    #[Computed]
    public function clinics()
    {
        return Clinic::where('status', 'active')
            ->orderBy('name')
            ->get();
    }

    #[Computed]
    public function filteredClinics()
    {
        if (empty($this->clinicSearch)) {
            return $this->clinics;
        }

        return $this->clinics->filter(function ($clinic) {
            return str_contains(strtolower($clinic->name), strtolower($this->clinicSearch));
        });
    }

    public function render()
    {
        return view('livewire.admin.doctor.doctor-form');
    }

    public function createDoctor()
    {
        Gate::authorize('create doctors');
        $this->validate();

        $valid = !$this->selected_clinics || collect($this->selected_clinics)->every(function ($clinicId) {
            return $this->checkDoctorLimits($clinicId);
        });
        if (!$valid) {
            return;
        }
        try {
            DB::transaction(function () {
                $user = $this->createUser();
                $this->assignDoctorRole($user);
                $this->createDoctorRecord($user);
                $this->attachClinicsToDoctor($user);

                $this->handleSuccessfulCreation($user);

                #Mail::to($user->email)->send(new CreateDoctorMail());
                #Log::info("✅ CreateDoctorMail sent to {$user->email}");
            });

        } catch (\Exception $e) {
            Log::error('Doctor creation failed: ' . $e->getMessage());
            notyf()->error('Failed to create doctor: ' . $e->getMessage());
        }
    }

    protected function createUser()
    {
        return User::create([
            'name' => $this->name,
            'email' => $this->email,
            'password' => Hash::make($this->password),
            'status' => $this->status,
        ]);
    }

    protected function assignDoctorRole(User $user)
    {
        $doctorRole = Role::where('name', 'doctor')
            ->where('guard_name', 'doctor')
            ->firstOrFail();
        $user->assignRole($doctorRole);
    }

    protected function createDoctorRecord(User $user)
    {
        $profileImagePath = $this->handleProfileImageUpload();

        Doctor::create([
            'user_id' => $user->id,
            'specialization' => $this->specialization,
            'qualifications' => $this->qualifications,
            'experience_years' => $this->experience_years,
            'license_number' => $this->license_number,
            'bio' => $this->bio,
            'profile_image' => $profileImagePath,
        ]);
    }

    protected function handleProfileImageUpload()
    {
        if ($this->profile_image) {

            //dd($this->profile_image);
            try {
                // Store the image in storage/app/public/doctor-profiles directory
                $path = $this->profile_image->store('doctor-profiles', 'public');
                Log::info('Profile image uploaded successfully', ['path' => $path]);
                return $path;
            } catch (\Exception $e) {
                Log::error('Failed to upload profile image: ' . $e->getMessage());
                // Don't throw exception, just return null so doctor creation doesn't fail
                return null;
            }
        }

        return null;
    }

    protected function handleSuccessfulCreation(User $user)
    {
        if ($this->isEmbedded && $this->clinicForm) {
            $this->dispatch('doctorCreated', $user->id);
            notyf()->success('Doctor created successfully.');
            return;
        }
        $this->redirect(route('admin.doctor-management'));
    }

    public function updateDoctor()
    {
        Gate::authorize('edit doctors');
        $this->validate($this->getUpdateRules());

        if($this->selected_clinics){
            $valid = collect($this->selected_clinics)->every(function ($clinicId) {
                $alreadyInClinic = DoctorClinic::where('clinic_id', $clinicId)->where('doctor_id' , $this->editId)->exists();
                if($alreadyInClinic){
                    return true;
                }
                return $this->checkDoctorLimits($clinicId);
            });
        }
        if(!$valid){
            return ;
        }

        try {
            DB::transaction(function () {
                $user = User::findOrFail($this->editId);
                $this->updateUser($user);
                $this->updateDoctorRecord($user);
                $this->syncClinicsForDoctor($user);

                #Mail::to($user->email)->send(new UpdateDoctorMail());
                #Log::info("✅ UpdateDoctorMail sent to {$user->email}");
            });

            notyf()->success('Doctor updated successfully.');
            return redirect()->route('admin.doctor-management.edit', ['id' => $this->editId]);

        } catch (\Exception $e) {
            Log::error('Doctor update failed: ' . $e->getMessage());
            notyf()->error('Failed to update doctor: ' . $e->getMessage());
        }
    }

    protected function getUpdateRules()
    {
        $rules = [
            'name' => 'required|string|max:255',
            'email' => [
                'required',
                'email',
                'max:255',
                Rule::unique('users', 'email')->ignore($this->editId)
            ],
            'specialization' => 'required|string|max:255',
            'qualifications' => 'nullable|string',
            'experience_years' => 'nullable|integer|min:0',
            'license_number' => [
                'required',
                'string',
                'max:255',
                Rule::unique('doctors', 'license_number')->ignore($this->editId, 'user_id')
            ],
            'bio' => 'nullable|string',
            'status' => 'required|in:active,inactive',
            'selected_clinics' => 'nullable|array',
            'selected_clinics.*' => 'exists:clinics,id',
            'profile_image' => 'nullable|image|max:2048',
        ];

        if ($this->password) {
            $rules['password'] = 'nullable|min:6|confirmed';
        }

        return $rules;
    }

    protected function updateUser(User $user)
    {
        $user->update([
            'name' => $this->name,
            'email' => $this->email,
            'status' => $this->status,
            'password' => $this->password ? Hash::make($this->password) : $user->password,
        ]);
    }

    protected function updateDoctorRecord(User $user)
    {
        $profileImagePath = $this->handleProfileImageUpdate($user);

        Doctor::updateOrCreate(
            ['user_id' => $user->id],
            [
                'specialization' => $this->specialization,
                'qualifications' => $this->qualifications,
                'experience_years' => $this->experience_years,
                'license_number' => $this->license_number,
                'bio' => $this->bio,
                'profile_image' => $profileImagePath,
            ]
        );
    }

    protected function handleProfileImageUpdate(User $user)
    {
        if ($this->profile_image) {
            // Delete old image if exists
            if ($this->existing_profile_image && Storage::disk('public')->exists($this->existing_profile_image)) {
                Storage::disk('public')->delete($this->existing_profile_image);
                Log::info('Old profile image deleted', ['path' => $this->existing_profile_image]);
            }

            // Upload new image
            try {
                $path = $this->profile_image->store('doctor-profiles', 'public');
                Log::info('Profile image updated successfully', ['path' => $path]);
                return $path;
            } catch (\Exception $e) {
                Log::error('Failed to update profile image: ' . $e->getMessage());
                // Return existing image path if upload fails
                return $this->existing_profile_image;
            }
        }

        // Return existing image if no new image uploaded
        return $this->existing_profile_image;
    }

    public function removeProfileImage()
    {
        if ($this->existing_profile_image && Storage::disk('public')->exists($this->existing_profile_image)) {
            Storage::disk('public')->delete($this->existing_profile_image);
            Log::info('Profile image removed', ['path' => $this->existing_profile_image]);
        }

        $this->existing_profile_image = null;
        $this->profile_image = null;

        // Update database
        if ($this->editId) {
            $user = User::findOrFail($this->editId);
            if ($user->doctor) {
                $user->doctor->update(['profile_image' => null]);
            }
        }

        notyf()->success('Profile image removed successfully.');
    }

    public function getProfileImageUrl()
    {
        // For new uploads (temporary files)
        if ($this->profile_image && is_object($this->profile_image)) {
            try {
                return $this->profile_image->temporaryUrl();
            } catch (\Exception $e) {
                Log::error('Failed to get temporary URL: ' . $e->getMessage());
                return null;
            }
        }

        // For existing images stored in database
        if ($this->existing_profile_image) {
            // Check if file exists in storage
            if (Storage::disk('public')->exists($this->existing_profile_image)) {
                $url = asset('storage/' . $this->existing_profile_image);
                Log::info('Profile image URL generated', [
                    'path' => $this->existing_profile_image,
                    'url' => $url,
                    'file_exists' => Storage::disk('public')->exists($this->existing_profile_image)
                ]);
                return $url;
            } else {
                Log::warning('Profile image file not found', [
                    'path' => $this->existing_profile_image,
                    'full_path' => storage_path('app/public/' . $this->existing_profile_image)
                ]);
                return null;
            }
        }

        return null;
    }

    // Add this method for debugging
    public function debugProfileImage()
    {
        Log::info('Debug Profile Image', [
            'existing_profile_image' => $this->existing_profile_image,
            'profile_image' => $this->profile_image ? 'New file uploaded' : 'No new file',
            'storage_path' => storage_path('app/public/'),
            'public_path' => public_path('storage/'),
            'storage_link_exists' => is_link(public_path('storage')),
            'file_exists' => $this->existing_profile_image ? Storage::disk('public')->exists($this->existing_profile_image) : false
        ]);
    }

    protected function checkDoctorLimits($clinicId)
    {
        $selectedClinic = Clinic::findOrFail($clinicId);
        if (!$selectedClinic->activeSubscription()) {
            session()->flash('error', "Failed to create doctor: No Active Subscription Plan For {$selectedClinic->name}.");
            notyf()->error("{$selectedClinic->name} does not have an Active Subscription Plan !");
            return false;
        } else {
            $activeSubscription = $selectedClinic->activeSubscription();
            $subscriptionPlan = $activeSubscription->plan;
            $totalDoctorsUnderClinic = DoctorClinic::where('clinic_id', $clinicId)->where('status' , 'active')->count();
            
            // Check if current doctor count is below the maximum allowed
            if ($totalDoctorsUnderClinic >= $subscriptionPlan->max_doctors) {
                session()->flash('error', "Failed to create doctor: {$selectedClinic->name} has reached the maximum doctors allowed under it's current plan. Upgrade now to add more.");
                notyf()->error("{$selectedClinic->name} has reached doctor limit of {$subscriptionPlan->max_doctors}.");
                return false;
            }
        } 

        return true;
    }

    protected function attachClinicsToDoctor(User $user)
    {
        if (empty($this->selected_clinics)) {
            return;
        }

        $clinicsToAttach = [];
        foreach ($this->selected_clinics as $clinicId) {
            $clinicsToAttach[$clinicId] = $this->getDefaultClinicPivotData();
        }

        $user->clinics()->attach($clinicsToAttach);
    }

    protected function syncClinicsForDoctor(User $user)
    {
        $existingPrimaryClinicId = $this->getExistingPrimaryClinicId($user->id);
        $clinicsToSync = $this->buildClinicsToSync($existingPrimaryClinicId);

        // Preserve primary clinic if it's not in selected clinics
        if ($existingPrimaryClinicId && !in_array($existingPrimaryClinicId, $this->selected_clinics)) {
            $clinicsToSync[$existingPrimaryClinicId] = array_merge(
                $this->getDefaultClinicPivotData(),
                ['is_primary' => true]
            );

            Log::info('Primary clinic preserved during doctor update', [
                'doctor_id' => $user->id,
                'primary_clinic_id' => $existingPrimaryClinicId,
                'selected_clinics' => $this->selected_clinics
            ]);
        }

        $user->clinics()->sync($clinicsToSync);
    }

    protected function getExistingPrimaryClinicId($doctorId)
    {
        return DoctorClinic::where('doctor_id', $doctorId)
            ->where('is_primary', true)
            ->value('clinic_id');
    }

    protected function buildClinicsToSync($existingPrimaryClinicId)
    {
        $clinicsToSync = [];

        foreach ($this->selected_clinics as $clinicId) {
            $isPrimary = $existingPrimaryClinicId && $clinicId == $existingPrimaryClinicId;
            
            $clinicsToSync[$clinicId] = array_merge(
                $this->getDefaultClinicPivotData(),
                ['is_primary' => $isPrimary]
            );
        }

        return $clinicsToSync;
    }

    protected function getDefaultClinicPivotData()
    {
        return [
            'is_primary' => false,
            'status' => 'active',
            'approval_status' => 'approved',
            'added_by' => auth()->guard('superadmin')->user()->id,
            'approved_at' => now(),
            'in_person_duration' => 15,
            'video_duration' => 15,
            'home_visit_duration' => 30,
            'offers_in_person' => true,
            'offers_video' => true,
            'offers_home_visit' => false,
        ];
    }

    public function getPrimaryClinicName()
    {
        if (!$this->primary_clinic_id) {
            return '';
        }

        $clinic = $this->clinics->where('id', $this->primary_clinic_id)->first();
        return $clinic ? $clinic->name : '';
    }

    public function updatedSelectedClinics($value)
    {
        // Ensure primary clinic is always selected
        if ($this->has_primary_clinic && 
            !empty($this->primary_clinic_id) &&
            !in_array($this->primary_clinic_id, $this->selected_clinics)) {
            $this->selected_clinics[] = $this->primary_clinic_id;
        }
    }

    #[Computed]
    public function selectedClinicsNames()
    {
        $selectedNames = [];
        foreach ($this->selected_clinics as $id) {
            $clinic = $this->clinics->firstWhere('id', $id);
            if ($clinic) {
                $name = $clinic->name;
                if ($this->has_primary_clinic && $id === $this->primary_clinic_id) {
                    $name .= ' (Primary)';
                }
                $selectedNames[] = $name;
            }
        }
        
        return $selectedNames;
    }
}