<?php

namespace App\Livewire\Admin\Doctor;

use App\Models\User;
use App\Models\Clinic;
use Livewire\Component;
use App\Models\DoctorClinic;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class DoctorClinicManager extends Component
{
    public $doctorId;
    public $selectedDoctorId;
    public $selectedClinicId;
    public $isPrimary = false;
    public $status = 'active';

    #[Validate('nullable|numeric|min:0')]
    public $inPersonFee;

    #[Validate('nullable|numeric|min:0')]
    public $videoFee;

    #[Validate('nullable|numeric|min:0')]
    public $homeVisitFee;

    public $showForm = false;
    public $editingAssociationId = null;

    protected $rules = [
        'selectedDoctorId' => 'required|exists:users,id',
        'selectedClinicId' => 'required|exists:clinics,id',
        'inPersonFee' => 'nullable|numeric|min:0',
        'videoFee' => 'nullable|numeric|min:0',
        'homeVisitFee' => 'nullable|numeric|min:0',
    ];

    public function mount($doctorId = null)
    {
        $this->selectedDoctorId = $doctorId;
        $this->doctorId = $doctorId;
    }

    public function updatedSelectedDoctorId()
    {
        $this->doctorId = $this->selectedDoctorId;
        $this->resetForm();
        $this->showForm = false;
    }

    public function render()
    {
        $doctors = User::has('doctor')
            ->with('doctor')
            ->orderBy('name')
            ->get();

        $doctor = null;
        $availableClinics = collect();
        $doctorAssociations = collect();

        if ($this->doctorId) {
            $doctor = User::with(['clinics', 'ownedClinics'])->find($this->doctorId);

            if ($doctor) {
                $availableClinics = Clinic::whereNotIn(
                    'id',
                    $doctor->clinics->pluck('id')
                )->get();

                $doctorAssociations = DoctorClinic::with(['clinic', 'addedBy'])
                    ->where('doctor_id', $this->doctorId)
                    ->get();
            }
        }

        return view('livewire.admin.doctor.doctor-clinic-manager', [
            'doctors' => $doctors,
            'doctor' => $doctor,
            'availableClinics' => $availableClinics,
            'doctorAssociations' => $doctorAssociations
        ]);
    }

    public function showAddForm()
    {
        if (!$this->doctorId) {
            notyf()->error('Please select a doctor first.');
            return;
        }

        $this->resetForm();
        $this->showForm = true;
    }

    public function editAssociation($associationId)
    {
        $association = DoctorClinic::find($associationId);

        if ($association && $association->doctor_id == $this->doctorId) {
            $this->editingAssociationId = $associationId;
            $this->selectedClinicId = $association->clinic_id;
            $this->isPrimary = $association->is_primary;
            $this->status = $association->status;
            $this->inPersonFee = $association->in_person_fee;
            $this->videoFee = $association->video_fee;
            $this->homeVisitFee = $association->home_visit_fee;
            $this->showForm = true;
        }
    }

    public function save()
    {
        $this->validate();

        if (!$this->doctorId) {
            notyf()->error('Please select a doctor first.');
            return;
        }

        try {
            DB::transaction(function () {

                if ($this->isPrimary) {

                    $existingPrimary = DoctorClinic::where('clinic_id', $this->selectedClinicId)
                        ->where('is_primary', true)
                        ->first();

                    if ($existingPrimary && $existingPrimary->doctor_id != $this->doctorId) {


                        $this->dispatch('confirm-primary-change', [
                            'currentDoctor' => User::find($existingPrimary->doctor_id)->name,
                            'newDoctor' => User::find($this->doctorId)->name,
                            'clinic' => Clinic::find($this->selectedClinicId)->name,
                            'clinicId' => $this->selectedClinicId,
                            'associationId' => $this->editingAssociationId
                        ]);
                        return;
                    }
                }

                $this->proceedWithSave();
            });
        } catch (\Exception $e) {
            Log::error('Error in save transaction', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            notyf()->error('Error: ' . $e->getMessage());
        }
    }

    public function proceedWithSave($clinicId = null, $isEditing = false)
    {
        $clinicId = $clinicId ?? $this->selectedClinicId;


        if (!$clinicId) {
            Log::error('Clinic ID is null in proceedWithSave');
            notyf()->error('Clinic ID is required.');
            return;
        }

        try {
            DB::transaction(function () use ($clinicId, $isEditing) {

                if ($this->isPrimary) {

                    // Remove primary status from any other clinics this doctor has
                    DoctorClinic::where('doctor_id', $this->doctorId)
                        ->where('is_primary', true)
                        ->update(['is_primary' => false]);

                    // Remove primary status from any other doctors at this clinic
                    DoctorClinic::where('clinic_id', $clinicId)
                        ->where('is_primary', true)
                        ->update(['is_primary' => false]);

                }

                $data = [
                    'doctor_id' => $this->doctorId,
                    'clinic_id' => $clinicId,
                    'is_primary' => $this->isPrimary,
                    'status' => $this->status,
                    'in_person_fee' => $this->inPersonFee,
                    'video_fee' => $this->videoFee,
                    'home_visit_fee' => $this->homeVisitFee,
                    'added_by' => auth()->id(),
                    'approval_status' => 'approved'
                ];


                if ($isEditing || $this->editingAssociationId) {
                    $associationId = $this->editingAssociationId;

                    DoctorClinic::where('id', $associationId)
                        ->update($data);
                    notyf()->success('Feature updated successfully.');
                    notyf()->success('Clinic association updated successfully!');
                } else {
                    DoctorClinic::create($data);
                    notyf()->success('Doctor added to clinic successfully!');
                }
            });

            $this->resetForm();
            $this->showForm = false;
        } catch (\Exception $e) {
            Log::error('Error in proceedWithSave', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            notyf()->error('Error saving clinic association: ' . $e->getMessage());
        }
    }
    public function removeAssociation($associationId)
    {
        try {
            $association = DoctorClinic::find($associationId);

            if ($association && $association->doctor_id == $this->doctorId) {
                $association->delete();
                notyf()->success('Clinic association removed successfully!');
            }
        } catch (\Exception $e) {
            notyf()->error('Error removing association: ' . $e->getMessage());
        }
    }

    public function setPrimary($associationId)
    {

        try {
            $association = DoctorClinic::findOrFail($associationId);

            // Check if clinic already has a primary doctor
            $existingPrimary = DoctorClinic::where('clinic_id', $association->clinic_id)
                ->where('is_primary', true)
                ->where('doctor_id', '!=', $this->doctorId)
                ->first();

            if ($existingPrimary) {

                $this->dispatch(
                    'confirm-primary-change-set',
                    currentDoctor: User::find($existingPrimary->doctor_id)->name,
                    newDoctor: User::find($this->doctorId)->name,
                    clinic: Clinic::find($association->clinic_id)->name,
                    clinicId: $association->clinic_id,
                    associationId: $associationId // Make sure this is passed correctly
                );
                return;
            }

            $this->proceedWithSetPrimary($associationId);
        } catch (\Exception $e) {
            Log::error('Error in setPrimary', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            notyf()->error('Error updating primary clinic: ' . $e->getMessage());
        }
    }

    public function proceedWithSetPrimary($associationId)
    {

        if (!$associationId) {
            notyf()->error('Invalid association ID');
            return;
        }

        try {
            $association = DoctorClinic::findOrFail($associationId);
            DB::transaction(function () use ($association) {
                DoctorClinic::where('doctor_id', $this->doctorId)
                    ->where('is_primary', true)
                    ->update(['is_primary' => false]);

                DoctorClinic::where('clinic_id', $association->clinic_id)
                    ->where('is_primary', true)
                    ->update(['is_primary' => false]);

                $association->update(['is_primary' => true]);
            });

            notyf()->success('Primary clinic updated successfully!');
        } catch (\Exception $e) {
            Log::error('Error in proceedWithSetPrimary', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            notyf()->error('Error updating primary clinic: ' . $e->getMessage());
        }
    }


    public function cancel()
    {
        $this->resetForm();
        $this->showForm = false;
    }

    private function resetForm()
    {
        $this->selectedClinicId = null;
        $this->isPrimary = false;
        $this->status = 'active';
        $this->inPersonFee = null;
        $this->videoFee = null;
        $this->homeVisitFee = null;
        $this->editingAssociationId = null;
        $this->resetErrorBag();
    }
}