<?php

namespace App\Livewire\Admin\DeviceTracking;

use App\Models\User;
use Livewire\Component;
use App\Models\ActiveDevice;
use Livewire\WithPagination;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Pagination\LengthAwarePaginator;

class DeviceTracking extends Component
{
    use WithPagination;

    public $search = '';

    public $users = [];
    public $selectedUserId = null;
    public $platformFilter = 'all';
    public $statusFilter = 'all';
    public $perPage = 10;
    public $sortField = 'last_active_at';
    public $sortDirection = 'desc';

    protected $listeners = ['deviceLoggedOut' => 'polling'];

    public function resetFilters()
    {
        $this->reset(['platformFilter', 'statusFilter', 'search']);
        if (!Auth::user()->is_superadmin) {
            $this->selectedUserId = Auth::id();
        }
    }

    public function mount()
    {
        $this->users = User::whereHas('roles', function ($query) {
            $query->whereIn('name', ['superadmin', 'superadminstaff']);
        })->get(['id', 'name', 'email']);

        if (!Auth::user()->is_superadmin) {
            $this->selectedUserId = Auth::id();
        }
    }

    public function getUserName($userId)
    {
        $user = collect($this->users)->firstWhere('id', $userId);
        return $user ? $user['name'] : 'Unknown User';
    }

    public function getDevices()
    {
        $userId = $this->selectedUserId ?: Auth::id();
        $currentDeviceId = Session::get('device_id');

        return ActiveDevice::where('user_id', $userId)
            ->where('is_active', 1)
            ->get()
            ->map(function ($device) use ($currentDeviceId) {
                $deviceArray = $device->toArray();
                $deviceArray['is_current'] = $device->device_id === $currentDeviceId;
                return $deviceArray;
            })
            ->toArray();
    }

    public function getFilteredDevicesProperty()
    {
        $devices = collect($this->getDevices());

        // Apply platform filter
        if ($this->platformFilter !== 'all') {
            $devices = $devices->filter(function ($device) {
                return strtolower($device['platform'] ?? 'web') === strtolower($this->platformFilter);
            });
        }

        // Apply status filter
        if ($this->statusFilter !== 'all') {
            $devices = $devices->filter(function ($device) {
                return $this->statusFilter === 'current' ? $device['is_current'] : !$device['is_current'];
            });
        }

        // Apply search filter across all fields
        if (!empty($this->search)) {
            $searchTerm = strtolower($this->search);
            $devices = $devices->filter(function ($device) use ($searchTerm) {
                // Search in device name
                if (str_contains(strtolower($device['device_name'] ?? ''), $searchTerm)) {
                    return true;
                }

                // Search in user name (if superadmin)
                if (Auth::user()->is_superadmin && str_contains(strtolower($this->getUserName($device['user_id'])), $searchTerm)) {
                    return true;
                }

                // Search in platform
                if (str_contains(strtolower($device['platform'] ?? ''), $searchTerm)) {
                    return true;
                }

                // Search in IP address
                if (str_contains(strtolower($device['ip_address'] ?? ''), $searchTerm)) {
                    return true;
                }

                // Search in login time (formatted)
                $loginTime = \Carbon\Carbon::parse($device['login_at'])->format('M d, Y H:i');
                if (str_contains(strtolower($loginTime), $searchTerm)) {
                    return true;
                }

                // Search in last active time
                $lastActive = \Carbon\Carbon::parse($device['last_active_at'])->diffForHumans();
                if (str_contains(strtolower($lastActive), $searchTerm)) {
                    return true;
                }

                // Search in status
                $status = $device['is_current'] ? 'active current' : 'idle';
                if (str_contains($status, $searchTerm)) {
                    return true;
                }

                return false;
            });
        }

        // Sort the results
        $devices = $devices->sortBy([
            [$this->sortField, $this->sortDirection]
        ]);

        // Paginate the results
        $currentPage = LengthAwarePaginator::resolveCurrentPage();
        $items = $devices->slice(($currentPage - 1) * $this->perPage, $this->perPage)->values();

        return new LengthAwarePaginator(
            $items,
            $devices->count(),
            $this->perPage,
            $currentPage,
            ['path' => request()->url(), 'query' => request()->query()]
        );
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
    }

    public function logoutDevice($deviceId)
    {
        $userId = $this->selectedUserId ?: Auth::id();
        $currentDeviceId = Session::get('device_id');

        if ($deviceId === $currentDeviceId && $userId !== Auth::id()) {
            $this->dispatch('notify', type: 'info', message: 'You cannot log out the current device of another user.');
            return;
        }

        $device = ActiveDevice::where('user_id', $userId)
            ->where('device_id', $deviceId)
            ->first();

        if ($device) {
            $device->update([
                'is_active' => false,
                'logout_at' => now(),
            ]);

            $sessionId = $device->session_id;
            DB::table('sessions')->where('id', $sessionId)->delete();
        }

        $this->dispatch('notify', type: 'success', message: 'Device logged out successfully');
        $this->polling();
    }

    public function logoutAllDevices()
    {
        $userId = Auth::id();
        $currentDeviceId = Session::get('device_id');

        ActiveDevice::where('user_id', $userId)
            ->where('device_id', '!=', $currentDeviceId)
            ->update(['is_active' => false, 'logout_at' => now()]);

        Auth::logout();
        session()->invalidate();
        session()->regenerateToken();

        ActiveDevice::where('user_id', $userId)
            ->where('device_id', $currentDeviceId)
            ->update(['is_active' => false, 'logout_at' => now()]);

        return redirect()->route('admin.login')->with('message', 'Successfully logged out from all devices');
    }

    public function polling()
    {
        $this->resetPage();
    }

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function render()
    {
        return view('livewire.admin.device-tracking.device-tracking');
    }
}
