<?php

namespace App\Livewire\Admin\Clinics;

use App\Models\Clinic;
use App\Models\User;
use App\Models\PatientClinic;
use Livewire\Component;
use Illuminate\Support\Facades\Gate;
use Livewire\Attributes\On;
use Livewire\WithPagination;

class PatientsTable extends Component
{
    use WithPagination;

    public Clinic $clinic;
    public $sortField = 'name';
    public $sortDirection = 'asc';
    public $isLoaded = false;
    public $shouldLoad = false;
    public $searchTerm = '';
    public $statusFilter = 'all';
    public $approvalStatusFilter = 'all';
    public $perPage = 10;
    public $deleteId = null;
    public $patientIdToDelete;
    public $clinicIdToDelete;
    protected $listeners = [
        'confirm-delete-patient' => 'confirmDeletePatient',
    ];

    public function updatedSearchTerm()
    {
        $this->resetPage();
    }
    public function openDeleteModal($patientId, $clinicId)
    {
        $this->dispatch(
            'show-shared-delete-modal',
            title: 'Delete Patient',
            message: 'Are you sure you want to remove this patient from the clinic?',
            patientId: $patientId,
            clinicId: $clinicId,
            action: 'confirm-delete-patient'
        );
    }
    public function confirmDeletePatient($patientId, $clinicId)
    {
        if (!$patientId || !$clinicId) {
            $this->dispatch('alert', type: 'error', message: 'Invalid patient or clinic.');
            return;
        }

        $this->deletePatient($patientId, $clinicId);
        $this->dispatch('hide-delete-modal');
    }


    public function applyFilters()
    {
        $this->resetPage();
    }

    public function clearAllFilters()
    {
        $this->statusFilter = 'all';
        $this->approvalStatusFilter = 'all';
        $this->searchTerm = '';
        $this->resetPage();
    }

    public function resetStatusFilter()
    {
        $this->statusFilter = 'all';
        $this->resetPage();
    }

    public function resetAprovelStatusFilter()
    {
        $this->approvalStatusFilter = 'all';
        $this->resetPage();
    }

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function mount(Clinic $clinic, $loadImmediately = false)
    {
        $this->clinic = $clinic;
        $this->shouldLoad = $loadImmediately;

        if ($this->shouldLoad) {
            $this->loadData();
        }
    }

    #[On('tabSwitched')]
    public function handleTabSwitch($tab)
    {
        if ($tab === 'patients' && !$this->isLoaded) {
            $this->loadData();
        }
    }

    public function loadData()
    {
        if (!$this->isLoaded) {
            $this->isLoaded = true;
        }
    }

    public function sortBy($field)
    {
        if (!$this->isLoaded) {
            $this->loadData();
        }

        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
        $this->resetPage();
    }

    public function togglePatientStatus($id, $clinicId, $doctorId)
    {
        $user = User::findOrFail($id);
        $association = PatientClinic::where('patient_id', $user->id)
            ->where('clinic_id', $clinicId)
            ->where('doctor_id', $doctorId)
            ->first();

        $association->status = $association->status === 'active' ? 'inactive' : 'active';
        $association->save();

        notyf()->success('Patient Clinic status updated successfully.');
    }

    public function togglePatientApprovalStatus($id, $clinicId, $doctorId, $status)
    {
        $user = User::findOrFail($id);
        $association = PatientClinic::where('patient_id', $user->id)
            ->where('clinic_id', $clinicId)
            ->where('doctor_id', $doctorId)
            ->first();

        $association->approval_status = $status;

        if ($status === 'rejected' || $status === 'pending') {
            $association->status = 'inactive';
        } elseif ($status === 'approved') {
            $association->status = 'active';
        }

        $association->save();
        notyf()->success('Patient Clinic Approval Status updated successfully.');
    }

    public function deletePatient($id, $clinicId)
    {
        Gate::authorize('delete patients');

        if (auth()->id() == $id) {
            $this->dispatch('alert', type: 'error', message: 'You cannot delete your own account.');
            return;
        }

        $user = User::findOrFail($id);

        if (!$user->hasRole('patient')) {
            $this->dispatch('alert', type: 'error', message: 'The selected user is not a patient.');
            return;
        }

        if (!$clinicId) {
            $this->dispatch('alert', type: 'error', message: 'No clinic selected.');
            return;
        }

        $association = PatientClinic::where('patient_id', $user->id)
            ->where('clinic_id', $clinicId)
            ->first();

        if ($association) {
            $association->delete();
            $this->deleteId = null;
        }

        notyf()->success('Patient removed from clinic successfully.');
    }

    public function render()
    {
        // Build the query with all filters and sorting
        $query = $this->clinic->patientAssociations()
            ->with(['patient', 'doctor'])
            ->join('users', 'patient_clinics.patient_id', '=', 'users.id')
            ->leftJoin('users as doctors', 'patient_clinics.doctor_id', '=', 'doctors.id');

        // Apply search filter
        if ($this->searchTerm) {
            $query->where(function ($subQuery) {
                $subQuery->where('users.name', 'like', '%' . $this->searchTerm . '%')
                    ->orWhere('users.email', 'like', '%' . $this->searchTerm . '%')
                    ->orWhere('doctors.name', 'like', '%' . $this->searchTerm . '%');
            });
        }

        // Apply status filter
        if ($this->statusFilter !== 'all') {
            $query->where('patient_clinics.status', $this->statusFilter);
        }

        // Apply approval status filter
        if ($this->approvalStatusFilter !== 'all') {
            $query->where('patient_clinics.approval_status', $this->approvalStatusFilter);
        }

        // Apply sorting
        switch ($this->sortField) {
            case 'name':
                $query->orderBy('users.name', $this->sortDirection);
                break;
            case 'email':
                $query->orderBy('users.email', $this->sortDirection);
                break;
            case 'status':
                $query->orderBy('patient_clinics.status', $this->sortDirection);
                break;
            case 'approval_status':
                $query->orderBy('patient_clinics.approval_status', $this->sortDirection);
                break;
            default:
                $query->orderBy('users.name', 'asc');
                break;
        }

        // Select the patient_clinics columns to avoid conflicts
        $query->select('patient_clinics.*');

        // Paginate the results
        $patientsAssociations = $query->paginate($this->perPage);

        return view('livewire.admin.clinics.patients-table', [
            'patients' => $patientsAssociations,
        ]);
    }
}