<?php

namespace App\Livewire\Admin\Clinics;

use App\Models\User;
use App\Models\Clinic;
use App\Models\Doctor;
use Livewire\Component;
use App\Models\DoctorClinic;
use App\Models\DoctorProfile;
use App\Models\SubscriptionPlan;
use App\Models\AddOn;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Log;
use App\Models\CustomerSubscription;
use App\Models\SubscriptionAddOn;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Hash;
use App\Mail\Clinic\CreateClinicMail;
use App\Mail\Clinic\UpdateClinicMail;
use App\Mail\SubscriptionMail;
use Illuminate\Support\Facades\Mail;

class ClinicForm extends Component
{
    public $name;
    public $email;
    public $owner_id;
    public $about;
    public $line1;
    public $line2;
    public $city;
    public $state;
    public $country;
    public $pincode;
    public $location;
    public $specialities = [];
    public $contact_numbers = [];
    public $status = 'active';
    public $clinic;
    public int $step = 1;
    public $showDoctorForm = false;
    public $tempDoctorData = null;
    public $newDoctorData = [
        'name' => '',
        'email' => '',
        'password' => '',
        'password_confirmation' => '',
        'specialization' => '',
        'license_number' => ''
    ];
    public $selected_plan_id = null;
    public $selected_add_on_id = [];
    public $showPlanSelection;
    public $showAddOnSelection;
    public $availablePlans;
    public $availableAddOns;
    public $showPlanSection = false;
    public $showAddOnSection = false;
    public $selectedPlanDetails = null;
    public $selectedPlanLimits = null;
    public $selectedAddOnDetails = [];
    public $editingDoctor = false;
    public $usedLimits = [];

    protected function getDoctorRules()
    {
        $rules = [
            'newDoctorData.name' => 'required|string|max:255',
            'newDoctorData.specialization' => 'required|string|max:255',
        ];

        if (!$this->editingDoctor) {
            $rules['newDoctorData.email'] = 'required|email|unique:users,email';
            $rules['newDoctorData.license_number'] = 'required|string|unique:doctors,license_number';
            $rules['newDoctorData.password'] = 'required|string|min:8|confirmed';
        } else {
            $rules['newDoctorData.email'] = 'required|email';
            $rules['newDoctorData.license_number'] = 'required|string';
            $rules['newDoctorData.password'] = 'nullable|string|min:8|confirmed';
        }

        return $rules;
    }

    protected $listeners = ['doctorCreated' => 'handleDoctorCreated'];
    public function togglePlanSelection()
    {
        $this->showPlanSelection = !$this->showPlanSelection;

        if ($this->selected_plan_id) {
            $this->selectedPlanDetails = SubscriptionPlan::find($this->selected_plan_id);
            $this->selectedPlanLimits = $this->selectedPlanDetails->limit;
            $this->loadUsedLimits();
        }
    }
    public function toggleAddOnSelection()
    {
        $this->showAddOnSelection = !$this->showAddOnSelection;

        if ($this->selected_add_on_id) {
            foreach ($this->selected_add_on_id as $index => $addOnId) {
                $this->selectedAddOnDetails[$index] = AddOn::find($addOnId);
            }
        } else {
            $this->selectedAddOnDetails = null;
        }
    }
    public function updatedSelectedPlanId($value)
    {
        if ($value) {
            $this->selectedPlanDetails = SubscriptionPlan::find($value);
            $this->showPlanSelection = false;
            $this->selectedPlanLimits = $this->selectedPlanDetails->limit;
            $this->loadUsedLimits();
        } else {
            $this->selectedPlanDetails = null;
        }
    }

    public function nextStep()
    {
        if ($this->step === 1) {
            $this->validate([
                'name' => 'required|string|max:255',
                'email' => 'nullable|email|max:255',
                'status' => 'required|in:active,inactive',
            ]);
            $this->step = 2;
        } elseif ($this->step === 2) {
            if ($this->showDoctorForm) {
                $this->validate($this->getDoctorRules());
                $this->tempDoctorData = $this->newDoctorData;
                $this->owner_id = null;
                $this->showDoctorForm = false;
            } else {
                $this->validate(['owner_id' => 'required|exists:users,id']);
                $this->tempDoctorData = null;
            }
            $this->step = 3;
        }
    }

    public function previousStep()
    {
        if ($this->step === 3) {
            $this->step = 2;
            if ($this->tempDoctorData) {
                $this->showDoctorForm = true;
                $this->newDoctorData = $this->tempDoctorData;
            }
        } elseif ($this->step === 2) {
            $this->step = 1;
            $this->tempDoctorData = null;
            $this->showDoctorForm = false;
        }
    }

    public function navigateToStep($step)
    {
        if($this->clinic->exists()){
            $this->step = $step;
        } else {
            $this->step++;
        }
    }

    public function save()
    {
        if ($this->clinic->exists) {
            Gate::authorize('edit clinics');
        } else {
            Gate::authorize('create clinics');
        }
        $this->validate();

        try {
            if ($this->clinic->exists) {
                $this->updateClinic();
            } else {
                $this->createClinic();
                notyf()->success('Clinic created successfully.');
                return redirect()->route('admin.clinics.index');
            }

            notyf()->success('Clinic saved successfully.');
        } catch (\Exception $e) {
            Log::error('Clinic save error: ' . $e->getMessage());
            notyf()->error('An error occurred: ' . $e->getMessage());
        }
    }

    private function createClinic()
    {
        $tempDoctorData = $this->tempDoctorData;

        DB::transaction(function () use ($tempDoctorData) {
            if ($tempDoctorData) {
                $this->createDoctorForClinic($tempDoctorData);
            }

            if ($this->owner_id) {
                $this->validateDoctorPrimaryStatus();
            }

            $data = $this->prepareClinicData();
            $this->clinic = Clinic::create($data);

            // ✅ Create clinic
            //$this->clinic = Clinic::create($data);

            // $qrData = 'Clinic ID: ' . $this->clinic->id . ' | Name: ' . $this->clinic->name;
            // $qrData = 'Clinic ID: ' . $this->clinic->id;
            // $qrData = $this->clinic->id;

            $qrData = [
                'clinic_id'   => $this->clinic->id,
                'clinic_name' => $this->clinic->name ?? null,
            ];

            // Convert array to JSON string ✅
            $qrString = json_encode($qrData);

            $fileName = 'qr_code_' . $this->clinic->id . '.svg';
            $path = 'qr_codes/' . $fileName;

            // Generate QR code from string ✅
            $qrImage = \SimpleSoftwareIO\QrCode\Facades\QrCode::format('svg')
                ->size(300)
                ->generate($qrString);

            \Illuminate\Support\Facades\Storage::disk('public')->put($path, $qrImage);

            // Save path in DB
            $this->clinic->update(['qr_code' => 'storage/' . $path]);



            // $qrData = [
            //     'clinic_id'   => $this->clinic->id,
            //     'clinic_name' => $this->clinic->name ?? null,
            // ];

            // // File name & path
            // $fileName = 'qr_code_' . $this->clinic->id . '.svg';
            // $path = 'qr_codes/' . $fileName;

            // // QR generate & save (json string pass करना होगा)
            // $qrImage = \SimpleSoftwareIO\QrCode\Facades\QrCode::format('svg')
            //     ->size(300)
            //     ->generate(json_encode($qrData));

            // \Illuminate\Support\Facades\Storage::disk('public')->put($path, $qrImage);

            // // Save path in DB
            // $this->clinic->update(['qr_code' => 'storage/' . $path]);

            // // Response return
            // return response()->json([
            //     'status' => 'success',
            //     'clinic_id' => $this->clinic->id,
            //     'qr_data' => $qrData, // array as JSON response
            //     'qr_code_url' => asset('storage/' . $path),
            // ]);


            // try {
            //     // ✅ API URL for QR Code (redirect ke liye)
            //     $qrData = route('clinic.qr.redirect', ['id' => $this->clinic->id]);

            //     // ✅ File name and path
            //     $fileName = 'qr_code_' . $this->clinic->id . '.svg';
            //     $path = 'qr_codes/' . $fileName;

            //     // ✅ Generate QR Code (SVG format)
            //     $qrImage = \SimpleSoftwareIO\QrCode\Facades\QrCode::format('svg')
            //         ->size(300)
            //         ->generate($qrData);

            //     // ✅ Save QR Code in storage/app/public/qr_codes
            //     \Illuminate\Support\Facades\Storage::disk('public')->put($path, $qrImage);

            //     // ✅ Build full URL dynamically using APP_URL
            //     $qrUrl = url('storage/' . $path);

            //     // ✅ Save only URL in DB
            //     $this->clinic->update([
            //         'qr_code' => $qrUrl
            //     ]);

            // } catch (\Exception $e) {
            //     \Log::error('QR Code Generation Failed: ' . $e->getMessage());
            // }

            // try {
            //     // ✅ API URL for QR Code
            //     $qrData = route('clinic.qr.redirect', ['id' => $this->clinic->id]);

            //     // ✅ File name and path
            //     $fileName = 'qr_code_' . $this->clinic->id . '.svg';
            //     $path = 'qr_codes/' . $fileName;

            //     // ✅ Generate QR Code (SVG format)
            //     $qrImage = \SimpleSoftwareIO\QrCode\Facades\QrCode::format('svg')
            //         ->size(300)
            //         ->generate($qrData);

            //     // ✅ Save QR Code in storage/app/public/qr_codes
            //     \Illuminate\Support\Facades\Storage::disk('public')->put($path, $qrImage);

            //     // ✅ Save full URL in database
            //     $this->clinic->update([
            //         'qr_code' => asset('storage/' . $path)
            //     ]);

            // } catch (\Exception $e) {
            //     \Log::error('QR Code Generation Failed: ' . $e->getMessage());
            // }

            if ($tempDoctorData) {
                $this->sendCreateClinicNotification($tempDoctorData);
            }

            if ($this->owner_id) {
                $this->createDoctorClinicRelation();
                $this->handleSubscriptionPlan();
                $this->handleAddOn();
            }
        });
    }

    private function updateClinic()
    {
        $tempDoctorData = $this->tempDoctorData;

        DB::transaction(function () use ($tempDoctorData) {
            if ($tempDoctorData) {
                if ($this->editingDoctor && $this->clinic->owner) {
                    $this->updateExistingDoctor($tempDoctorData);
                } else {
                    $this->createDoctorForClinic($tempDoctorData);
                }
            }

            if ($this->owner_id) {
                $this->validateDoctorPrimaryStatus();
            }

            $data = $this->prepareClinicData();
            $this->clinic->update($data);

            if ($this->owner_id) {
                $this->createDoctorClinicRelation();
                $this->handleSubscriptionPlan();
                $this->handleAddOn();
            } else {
                $this->removeSubscriptionIfNoOwner();
            }
        });
    }

    private function createDoctorForClinic($tempDoctorData)
    {
        $user = User::create([
            'name' => $tempDoctorData['name'],
            'email' => $tempDoctorData['email'],
            'password' => Hash::make($tempDoctorData['password']),
            'status' => 'active',
        ]);

        $this->assignDoctorRoles($user);
        $this->createDoctorProfiles($user, $tempDoctorData);

        $this->owner_id = $user->id;
    }

    private function updateExistingDoctor($tempDoctorData)
    {
        $user = $this->clinic->owner;
        $user->update([
            'name' => $tempDoctorData['name'],
            'email' => $tempDoctorData['email'],
            'status' => 'active',
        ]);


        if (!empty($tempDoctorData['password'])) {
            $user->update([
                'password' => Hash::make($tempDoctorData['password']),
            ]);
        }

        $user->doctorProfile()->updateOrCreate(
            ['user_id' => $user->id],
            [
                'specialization' => $tempDoctorData['specialization'],
                'license_number' => $tempDoctorData['license_number'],
            ]
        );

        $this->owner_id = $user->id;
    }

    private function assignDoctorRoles($user)
    {
        $doctorRoleWeb = Role::where('name', 'primary_doctor')->where('guard_name', 'web')->first();
        $doctorRoleDoctor = Role::where('name', 'primary_doctor')->where('guard_name', 'doctor')->first();

        if ($doctorRoleWeb) {
            $user->assignRole($doctorRoleWeb);
        }

        if ($doctorRoleDoctor) {
            $user->assignRole($doctorRoleDoctor);
        }
    }

    private function createDoctorProfiles($user, $tempDoctorData)
    {
        Doctor::create([
            'user_id' => $user->id,
            'specialization' => $tempDoctorData['specialization'],
            'license_number' => $tempDoctorData['license_number'],
        ]);

        DoctorProfile::create([
            'user_id' => $user->id,
            'specialization' => $tempDoctorData['specialization'],
            'license_number' => $tempDoctorData['license_number'],
        ]);
    }

    private function validateDoctorPrimaryStatus()
    {
        $alreadyPrimary = DoctorClinic::where('doctor_id', $this->owner_id)
            ->where('is_primary', true)
            ->when($this->clinic?->id, fn($q) => $q->where('clinic_id', '!=', $this->clinic->id))
            ->exists();

        if ($alreadyPrimary) {
            throw new \Exception('This doctor is already a primary doctor in another clinic.');
        }
    }

    private function prepareClinicData()
    {
        return [
            'name' => $this->name,
            'email' => $this->email,
            'owner_id' => $this->owner_id,
            'about' => $this->about,
            'line1' => $this->line1,
            'line2' => $this->line2,
            'city' => $this->city,
            'state' => $this->state,
            'country' => $this->country,
            'pincode' => $this->pincode,
            'location' => $this->location,
            'specialities' => !empty($this->specialities) ? $this->specialities : null,
            'contact_numbers' => !empty($this->contact_numbers) ? $this->contact_numbers : null,
            'status' => $this->status,
            'added_by' => auth()->user()->id,
        ];
    }

    private function sendCreateClinicNotification($tempDoctorData)
    {
        $doctorEmail = $tempDoctorData['email'];
        $temporaryPassword = $tempDoctorData['password'];

        try {
            Log::info("Sending CreateClinicMail to {$doctorEmail} with temporary password: {$temporaryPassword}");
            /* Mail::to($doctorEmail)->send(new CreateClinicMail(
            clinicAdminName: $tempDoctorData['name'],
            clinicName: $this->name,
            adminUsername: $tempDoctorData['email'],
            temporaryPassword: $temporaryPassword
        )); */
            Log::info("CreateClinicMail successfully sent to {$doctorEmail}");
        } catch (\Exception $e) {
            Log::error("Failed to send CreateClinicMail to {$doctorEmail}: " . $e->getMessage());
        }
    }

    private function createDoctorClinicRelation()
    {
        DoctorClinic::updateOrCreate(
            ['doctor_id' => $this->owner_id, 'clinic_id' => $this->clinic->id],
            [
                'is_primary' => true,
                'status' => 'active',
                'approval_status' => 'approved',
                'approved_at' => now(),
                'offers_in_person' => true,
                'offers_video' => true,
                'offers_home_visit' => false,
                'added_by' => auth()->id(),
            ]
        );
    }

    private function removeSubscriptionIfNoOwner()
    {
        if ($this->clinic->subscription_id) {
            $this->clearSubscriptionAddOns($this->clinic->subscription_id);
            CustomerSubscription::where('id', $this->clinic->subscription_id)->delete();
            $this->clinic->update(['subscription_id' => null]);
        }
    }

    protected function handleSubscriptionPlan()
    {
        if (!$this->selected_plan_id) {
            if ($this->clinic->subscription_id) {
                $this->clearSubscriptionAddOns($this->clinic->subscription_id);
                CustomerSubscription::where('id', $this->clinic->subscription_id)->delete();
                $this->clinic->update(['subscription_id' => null]);
            }
            return;
        }

        $plan = SubscriptionPlan::findOrFail($this->selected_plan_id);

        if ($this->clinic->subscription_id) {
            $currentSubscription = CustomerSubscription::find($this->clinic->subscription_id);

            if (!$currentSubscription || $currentSubscription->plan_id != $this->selected_plan_id) {
                $this->clearSubscriptionAddOns($this->clinic->subscription_id);
                $this->clinic->update(['subscription_id' => null]);
                if ($currentSubscription) {
                    $currentSubscription->delete();
                }
                $this->createNewSubscription($plan);
            } else {
                $currentSubscription->update([
                    'user_id' => $this->owner_id,
                    'status' => 'active',
                    'auto_renew' => true,
                ]);
            }
        } else {
            $this->createNewSubscription($plan);
        }
    }

    protected function clearSubscriptionAddOns($subscriptionId)
    {
        SubscriptionAddOn::where('subscription_id', $subscriptionId)->delete();
    }

    protected function createNewSubscription($plan)
    {
        $duration = (int) $plan->duration;

        $endDate = match ($plan->plan_type) {
            'monthly' => now()->addMonths($duration),
            'annual' => now()->addYears($duration),
            'custom' => now()->addMonths($duration),
            default => now()->addMonths($duration)
        };

        $subscription = CustomerSubscription::create([
            'subscription_code' => 'SUB-' . strtoupper(uniqid()),
            'user_id' => $this->owner_id,
            'plan_id' => $plan->id,
            'clinic_id' => $this->clinic->id,
            'start_date' => now(),
            'end_date' => $endDate,
            'renewal_date' => $endDate,
            'status' => 'active',
            'auto_renew' => true,
            'billing_cycle' => $plan->plan_type,
        ]);

        $this->clinic->update(['subscription_id' => $subscription->id]);
    }

    protected function handleAddOn()
    {
        if (!$this->clinic->subscription_id) {
            return;
        }

        if (empty($this->selected_add_on_id)) {
            $this->clearSubscriptionAddOns($this->clinic->subscription_id);
            return;
        }

        $addOns = AddOn::whereIn('id', $this->selected_add_on_id)->get();

        if (!$addOns->isEmpty()) {
            $this->clearSubscriptionAddOns($this->clinic->subscription_id);
            $this->createNewAddOn($addOns);
        } else {
            $this->clearSubscriptionAddOns($this->clinic->subscription_id);
        }
    }

    protected function createNewAddOn($addOns)
    {
        foreach ($addOns as $addOn) {
            SubscriptionAddOn::create([
                'subscription_id' => $this->clinic->subscription_id,
                'addon_id' => $addOn->id,
                'units_purchased' => 1,
                'total_amount' => $addOn->price_per_unit * $addOn->unit_counts * 1,
                'purchase_date' => now(),
            ]);
        }
    }

    public function showNewDoctorForm()
    {
        $this->showDoctorForm = true;
    }

    public function handleDoctorCreated($doctorId)
    {
        $this->owner_id = $doctorId;
        $this->showDoctorForm = false;
        $this->dispatch('owner-selected', $doctorId);
    }

    protected $rules = [
        'name' => 'required|string|max:255',
        'email' => 'nullable|email|max:255',
        'owner_id' => 'nullable|exists:users,id',
        'about' => 'nullable|string',
        'line1' => 'nullable|string|max:255',
        'line2' => 'nullable|string|max:255',
        'city' => 'nullable|string|max:100',
        'state' => 'nullable|string|max:100',
        'country' => 'nullable|string|max:100',
        'pincode' => 'nullable|string|max:20',
        'location' => 'nullable',
        'specialities' => 'nullable|array',
        'contact_numbers' => 'nullable|array',
        'status' => 'required|in:active,inactive',
    ];

    public function mount(Clinic $clinic = null)
    {
        if ($clinic->exists) {
            Gate::authorize('edit clinics');
            if ($clinic->owner) {
                $this->editingDoctor = true;
                $this->newDoctorData = [
                    'name' => $clinic->owner->name,
                    'email' => $clinic->owner->email,
                    'password' => '',
                    'password_confirmation' => '',
                    'specialization' => $clinic->owner->doctorProfile->specialization ?? '',
                    'license_number' => $clinic->owner->doctorProfile->license_number ?? ''
                ];
            }
        } else {
            Gate::authorize('create clinics');
        }
        $this->clinic = $clinic;
        $this->availablePlans = SubscriptionPlan::where('plan_status', 'active')->get();
        $this->availableAddOns = AddOn::where('is_active', true)->get();

        if ($this->clinic->exists) {
            $this->name = $clinic->name;
            $this->email = $clinic->email;
            $this->owner_id = $clinic->owner_id;
            $this->about = $clinic->about;
            $this->line1 = $clinic->line1;
            $this->line2 = $clinic->line2;
            $this->city = $clinic->city;
            $this->state = $clinic->state;
            $this->country = $clinic->country;
            $this->pincode = $clinic->pincode;
            $this->location = $clinic->location;
            $this->specialities = $clinic->specialities ?? [];
            $this->contact_numbers = $clinic->contact_numbers ?? [];
            $this->status = $clinic->status ?? 'active';

            // Load existing subscription if available
            if ($clinic->subscription_id) {
                $subscription = CustomerSubscription::find($clinic->subscription_id);
                $this->selected_plan_id = $subscription->plan_id;
                $this->selectedPlanDetails = $subscription->plan;
                $this->selectedPlanLimits = $this->selectedPlanDetails->limit;
                $this->loadUsedLimits();

                if ($clinic->subscription->subscriptionAddOns) {
                    foreach ($clinic->subscription->subscriptionAddOns as $index => $addOn) {
                        $addOn = $addOn->addOn;
                        $this->selected_add_on_id[$index] = $addOn->id;
                        $this->selectedAddOnDetails[$index] = $addOn;
                    }
                }
            }
        }
    }

    protected function loadUsedLimits()
    {
        if ($this->clinic->exists) {
            $this->usedLimits['doctors'] = $this->clinic->doctorAssociations->count();
            $this->usedLimits['staffs'] = $this->clinic->staffAssociations->count();
            $this->usedLimits['storage_limit'] = $this->selectedPlanLimits->storage_limit_gb;
            $this->usedLimits['teleconsultation_limit'] = $this->selectedPlanLimits->teleconsultation_minutes;
            $this->usedLimits['sms_limit'] = $this->selectedPlanLimits->sms_limit;
            $this->usedLimits['whatsapp_limit'] = $this->selectedPlanLimits->whatsapp_limit;
        } else {
            $this->usedLimits['doctors'] = 0;
            $this->usedLimits['staffs'] = 0;
            $this->usedLimits['storage_limit'] = 0;
            $this->usedLimits['teleconsultation_limit'] = 0;
            $this->usedLimits['sms_limit'] = 0;
            $this->usedLimits['whatsapp_limit'] = 0;
        }
    }

    public function addContactNumber()
    {
        $this->contact_numbers[] = '';
    }

    public function removeContactNumber($index)
    {
        unset($this->contact_numbers[$index]);
        $this->contact_numbers = array_values($this->contact_numbers);
    }

    public function addSpeciality()
    {
        $this->specialities[] = '';
    }

    public function removeSpeciality($index)
    {
        unset($this->specialities[$index]);
        $this->specialities = array_values($this->specialities);
    }

    public function updateSelectedCount()
    {
        // Just forces a UI refresh
        $this->dispatch('addons-updated');
        if ($this->selected_add_on_id) {
            $this->selectedAddOnDetails = [];
            foreach ($this->selected_add_on_id as $index => $addOnId) {
                $this->selectedAddOnDetails[$index] = AddOn::find($addOnId);
            }
        } else {
            $this->selectedAddOnDetails = null;
        }
    }

    public function render()
    {
        $allDoctors = User::role('doctor', 'doctor');
        $owners = $allDoctors->whereDoesntHave('clinicAssociations', function ($query) {
            $query->where('is_primary', true)
                ->when($this->clinic?->id, function ($q) {
                    $q->where('clinic_id', '!=', $this->clinic->id);
                });
        })->get();

        $selectedOwner = $this->owner_id ? User::find($this->owner_id) : null;

        return view('livewire.admin.clinics.clinic-form', [
            'owners' => $owners,
            'selectedOwner' => $selectedOwner
        ]);
    }
}
