<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use App\Services\DoctorClinicService;

class CheckClinicPermissions
{
    protected $clinicService;

    public function __construct(DoctorClinicService $clinicService)
    {
        $this->clinicService = $clinicService;
    }

    public function handle($request, Closure $next, $permission)
    {
        $user = Auth::guard('doctor')->user();
        $userId = $user ? $user->id : 'guest';

        Log::channel('permissions')->info('Permission check initiated', [
            'user_id' => $userId,
            'permission' => $permission,
            'route' => $request->route()->getName(),
            'clinic_id' => session('current_clinic_id')
        ]);

        if (!$user) {
            Log::channel('permissions')->warning('Unauthorized access attempt - no user');
            return response()->view('errors.403', [
                'error' => [
                    'message' => 'Unauthorized access',
                    'permission' => $permission
                ]
            ], 403);
        }

        // Bypass for superadmins
        if ($user->hasRole('superadmin')) {
            Log::channel('permissions')->info('Superadmin bypass', [
                'user_id' => $userId,
                'roles' => $user->getRoleNames()
            ]);
            return $next($request);
        }

        $clinicId = session('current_clinic_id');

        if (!$clinicId) {
            Log::channel('permissions')->warning('No clinic selected', [
                'user_id' => $userId,
                'session_data' => session()->all()
            ]);
            return response()->view('errors.403', [
                'error' => [
                    'message' => 'No clinic selected',
                    'permission' => $permission
                ]
            ], 403);
        }

        // Add this right before the Primary Clinic Permissiosn check
        Log::channel('permissions')->info("Clinic Role : " . session('clinic_role'));
        // Handle sub-doctor permissions
        if (session('clinic_role') === 'sub_doctor') {
            $pivotData = $user->doctorClinics()->where('clinic_id', $clinicId)->first()->pivot;
            Log::channel('permissions')->info("Clinic Pivote Data : " . $pivotData);

            if ($pivotData->override_default_permissions) {
                // First decode the JSON string to array
                $extraPermissions = json_decode($pivotData->extra_permissions, true) ?? [];

                // Sometimes the data might be double-encoded, so we decode again if needed
                if (is_string($extraPermissions)) {
                    $extraPermissions = json_decode($extraPermissions, true) ?? [];
                }

                if (in_array($permission, $extraPermissions)) {
                    Log::channel('permissions')->info('Permission granted via clinic-specific permissions', [
                        'user_id' => $user->id,
                        'clinic_id' => $clinicId,
                        'permission' => $permission,
                        'source' => 'session_role_and_pivot'
                    ]);
                    return $next($request);
                }

                Log::channel('permissions')->warning('Sub-doctor permission denied', [
                    'user_id' => $user->id,
                    'clinic_id' => $clinicId,
                    'permission' => $permission,
                    'extra_permissions' => $extraPermissions
                ]);

                return response()->view('errors.403', [
                    'error' => [
                        'message' => 'Unauthorized for this clinic action',
                        'permission' => $permission
                    ]
                ], 403);
            } else {
                Log::channel('permissions')->warning('Sub-doctor permission denied', [
                    'user_id' => $user->id,
                    'clinic_id' => $clinicId,
                    'permission' => $permission,
                    'override_default_permissions' => false,
                ]);

                return response()->view('errors.403', [
                    'error' => [
                        'message' => 'Unauthorized for this clinic action',
                        'permission' => $permission
                    ]
                ], 403);
            }
        }

        $association = $this->clinicService->getClinicAssociation($user, $clinicId);

        if (!$association) {
            Log::channel('permissions')->warning('Clinic association not found', [
                'user_id' => $userId,
                'clinic_id' => $clinicId,
                'available_clinics' => $this->clinicService->getAvailableClinics($user)
            ]);
            return response()->view('errors.403', [
                'error' => [
                    'message' => 'Not associated with this clinic',
                    'permission' => $permission
                ]
            ], 403);
        }

        if (str_contains($permission, 'staff')) {
            if (!$association['is_primary']) {
                Log::channel('permissions')->warning('Staff permission denied - not primary clinic', [
                    'user_id' => $user->id,
                    'clinic_id' => $clinicId,
                    'permission' => $permission
                ]);
                return response()->view('errors.403', [
                    'error' => [
                        'message' => 'Staff management only available in primary clinic',
                        'permission' => $permission
                    ]
                ], 403);
            }
            return $next($request);
        }

        $hasPermission = $user->hasPermissionTo($permission) ||
            in_array($permission, $association['permissions'] ?? []);

        if (!$hasPermission) {
            Log::channel('permissions')->warning('Permission denied', [
                'user_id' => $userId,
                'clinic_id' => $clinicId,
                'permission' => $permission,
                'user_permissions' => $user->getPermissionNames(),
                'clinic_permissions' => $association['permissions'] ?? [],
                'is_primary' => $association['is_primary']
            ]);
            return response()->view('errors.403', [
                'error' => [
                    'message' => 'Unauthorized for this clinic',
                    'permission' => $permission
                ]
            ], 403);
        }

        Log::channel('permissions')->info('Permission granted', [
            'user_id' => $userId,
            'clinic_id' => $clinicId,
            'permission' => $permission,
            'granted_via' => $user->hasPermissionTo($permission) ? 'global' : 'clinic'
        ]);

        return $next($request);
    }
}
