<?php

namespace App\Http\Controllers;

use Illuminate\Http\Client\RequestException;
use Illuminate\Support\Str;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Auth;
use Firebase\JWT\JWT;
use Firebase\JWT\Key;

class VideoSDKController extends Controller
{
    /** Build a short-lived JWT for VideoSDK REST + JS SDK */
    private function buildJwt(): string
    {
        $apiKey = env('VIDEOSDK_API_KEY');
        $secret = env('VIDEOSDK_SECRET_KEY');

        if (empty($apiKey) || empty($secret)) {
            throw new \RuntimeException('VIDEOSDK_API_KEY or VIDEOSDK_SECRET_KEY missing in .env');
        }

        $payload = [
            'apikey'      => $apiKey,
            'permissions' => ['allow_join', 'allow_mod'], // adjust if needed
            'iat'         => time(),
            //'exp'         => time() + 60 * 60,            // 1 hour
            //'exp' => time() + (60 * 60 * 4),   // 4 hours
            'exp' => time() + (60 * 60 * 24 * 7),
        ];

        return JWT::encode($payload, (string)$secret, 'HS256');
    }

    /** Optional token endpoint (kept for parity) */
    public function getToken(Request $request)
    {
        try {
            return response()->json(['token' => $this->buildJwt()]);
        } catch (\Throwable $e) {
            Log::error('VideoSDK token error', ['m' => $e->getMessage()]);
            return response()->json(['error' => 'Token generation error'], 500);
        }
    }

    /** Ensure table exists (quick bootstrap). Prefer a real migration in production. */
    private function ensureAppointmentsTable(): void
    {
        DB::statement("
            CREATE TABLE IF NOT EXISTS appointments (
              id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
              appointment_id VARCHAR(64) NOT NULL UNIQUE,
              patient_id BIGINT UNSIGNED DEFAULT 0,
              doctor_id  BIGINT UNSIGNED DEFAULT 0,
              meeting_id VARCHAR(64) DEFAULT NULL,
              token TEXT DEFAULT NULL,
              created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
              updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
        ");
    }

    /** Blade page; auto-append ?doctor_id= / ?patient_id= based on guard */
    public function videoCall(Request $request)
    {
        $role = null;
        $uid  = null;

        if (Auth::guard('doctor')->check()) {
            $role = 'doctor';
            $uid  = Auth::guard('doctor')->id();
        } elseif (Auth::guard('patient')->check()) {
            $role = 'patient';
            $uid  = Auth::guard('patient')->id();
        } elseif (Auth::guard('superadmin')->check()) {
            $role = 'admin';
            $uid  = Auth::guard('superadmin')->id();
        } else {
            abort(401, 'Unauthorized');
        }

        // Auto-append user id in query (?doctor_id= / ?patient_id=)
        $append = [];
        if ($role === 'doctor' && !$request->has('doctor_id')) {
            $append['doctor_id'] = $uid;
        }
        if ($role === 'patient' && !$request->has('patient_id')) {
            $append['patient_id'] = $uid;
        }

        if (!empty($append)) {
            return redirect()->to($request->fullUrlWithQuery($append));
        }

        // Pass variables to blade for JS to use
        $appointmentId = $request->query('appointment_id'); // you decide how you pass this
  

        // VideoSDKController.php
        $jsonEndpoint = match ($role) {
            'doctor' => route('doctor.videosdk.meeting', [], false), // <= relative: "/doctor/videosdk/meeting"
            'patient'=> route('patient.videosdk.meeting', [], false),
            default  => route('admin.videosdk.meeting',  [], false),
        };

        return view('video-conference', [
            'role'          => $role,
            'uid'           => $uid,
            'appointmentId' => $appointmentId,
            'jsonEndpoint'  => $jsonEndpoint,
        ]);
    }

    /** JSON create/fetch meeting (Laravel version of your meeting.php) */
    public function createOrFetchMeeting(Request $request)
    {
        $cid = (string) Str::uuid();               // correlation id for this request
        $t0  = microtime(true);

        try {
            $appointment_id = trim((string)$request->query('appointment_id', ''));
            if ($appointment_id === '') {
                Log::warning('videosdk.meeting.invalid', [
                    'cid' => $cid, 'reason' => 'missing appointment_id', 'q' => $request->query()
                ]);
                return response()
                    ->json(['error' => 'appointment_id is required', 'correlation_id' => $cid], 422)
                    ->header('X-Correlation-ID', $cid);
            }

            // Log who/what called us
            $who = [
                'doctor_guard'  => Auth::guard('doctor')->check() ? Auth::guard('doctor')->id() : null,
                'patient_guard' => Auth::guard('patient')->check() ? Auth::guard('patient')->id() : null,
                'super_guard'   => Auth::guard('superadmin')->check() ? Auth::guard('superadmin')->id() : null,
                'ip'            => $request->ip(),
                'ua'            => $request->userAgent(),
                'q'             => $request->query(),
            ];
            Log::info('videosdk.meeting.start', ['cid' => $cid, 'appointment_id' => $appointment_id] + $who);

            // 1) Find existing appointment (do NOT insert into this FK table)
            $tDb0 = microtime(true);
            $appt = DB::table('appointments')->where('appointment_id', $appointment_id)->first();
            $tDb1 = microtime(true);
            Log::debug('videosdk.meeting.db_lookup', [
                'cid' => $cid, 'ms' => (int)(($tDb1-$tDb0)*1000), 'found' => (bool)$appt
            ]);

            if (!$appt) {
                Log::warning('videosdk.meeting.not_found', ['cid' => $cid, 'appointment_id' => $appointment_id]);
                return response()
                    ->json(['error' => 'Invalid appointment_id. Appointment not found.', 'correlation_id' => $cid], 404)
                    ->header('X-Correlation-ID', $cid);
            }

            // 2) If meeting already exists
            if (!empty($appt->meeting_id)) {
                Log::info('videosdk.meeting.join_existing', ['cid' => $cid, 'meeting_id' => $appt->meeting_id]);
                return response()
                    ->json([
                        'status'     => 'join',
                        'meeting_id' => $appt->meeting_id,
                        'token'      => $appt->token ?: $this->buildJwt(),
                        'correlation_id' => $cid,
                    ])
                    ->header('X-Correlation-ID', $cid);
            }

            // 3) Create a room via VideoSDK
            $token = $this->buildJwt();
            $tHttp0 = microtime(true);
            try {
                $resp = Http::withHeaders([
                        'Authorization' => $token,
                        'Content-Type'  => 'application/json',
                    ])
                    ->timeout(20)        // prod timeout
                    ->retry(2, 250)      // quick retry
                    ->post('https://api.videosdk.live/v2/rooms', (object)[]);

                $tHttp1 = microtime(true);

                Log::info('videosdk.http.rooms', [
                    'cid'     => $cid,
                    'status'  => $resp->status(),
                    'ok'      => $resp->ok(),
                    'ms'      => (int)(($tHttp1-$tHttp0)*1000),
                    'body_snippet' => Str::limit($resp->body(), 500),
                ]);

                if (!$resp->ok()) {
                    return response()
                        ->json([
                            'error' => 'VideoSDK create room failed',
                            'status' => $resp->status(),
                            'raw'   => $resp->json(),
                            'correlation_id' => $cid
                        ], 500)
                        ->header('X-Correlation-ID', $cid);
                }

                $data       = $resp->json();
                $meeting_id = $data['roomId'] ?? $data['id'] ?? null;
                if (!$meeting_id) {
                    Log::error('videosdk.http.invalid_response', ['cid' => $cid, 'json' => $data]);
                    return response()
                        ->json(['error' => 'Invalid VideoSDK response', 'raw' => $data, 'correlation_id' => $cid], 500)
                        ->header('X-Correlation-ID', $cid);
                }
            } catch (RequestException $e) {
                // Captures network/SSL issues clearly
                Log::error('videosdk.http.request_exception', [
                    'cid'     => $cid,
                    'message' => $e->getMessage(),
                    'status'  => optional($e->response)->status(),
                    'body'    => optional($e->response)->body(),
                    'ctx'     => method_exists($e, 'getHandlerContext') ? $e->getHandlerContext() : null,
                ]);
                return response()
                    ->json(['error' => 'Network/SSL error contacting VideoSDK', 'message' => $e->getMessage(), 'correlation_id' => $cid], 500)
                    ->header('X-Correlation-ID', $cid);
            } catch (\Throwable $e) {
                Log::error('videosdk.http.generic_exception', ['cid' => $cid, 'message' => $e->getMessage()]);
                return response()
                    ->json(['error' => 'Unexpected error contacting VideoSDK', 'message' => $e->getMessage(), 'correlation_id' => $cid], 500)
                    ->header('X-Correlation-ID', $cid);
            }

            // 4) Update SAME appointment row (no insert)
            DB::table('appointments')->where('id', $appt->id)->update([
                'meeting_id' => $meeting_id,
                'token'      => $token,
                'updated_at' => now(),
            ]);

            Log::info('videosdk.meeting.created', ['cid' => $cid, 'meeting_id' => $meeting_id]);

            return response()
                ->json([
                    'status'        => 'new',
                    'meeting_id'    => $meeting_id,
                    'token'         => $token,
                    'correlation_id'=> $cid,
                ])
                ->header('X-Correlation-ID', $cid);

        } catch (\Throwable $e) {
            Log::error('videosdk.meeting.unhandled', [
                'cid' => $cid,
                'error' => $e->getMessage(),
                'trace' => Str::limit($e->getTraceAsString(), 2000),
            ]);

            return response()
                ->json(['error' => 'Server error', 'message' => $e->getMessage(), 'correlation_id' => $cid], 500)
                ->header('X-Correlation-ID', $cid);
        } finally {
            Log::debug('videosdk.meeting.end', ['cid' => $cid, 'ms' => (int)((microtime(true)-$t0)*1000)]);
        }
    }

    public function diagnostics()
    {
        $cid = (string) Str::uuid();
        try {
            $dns = @gethostbyname('api.videosdk.live');

            $http = Http::timeout(10)
                ->withOptions(['verify' => true]) // set to false ONLY to confirm CA issue
                ->get('https://api.videosdk.live');  // expect 404/401 but TLS must succeed

            $curl = function_exists('curl_version') ? curl_version() : null;

            $data = [
                'correlation_id' => $cid,
                'env' => [
                    'APP_ENV'   => env('APP_ENV'),
                    'APP_DEBUG' => env('APP_DEBUG'),
                ],
                'php' => [
                    'PHP_VERSION'   => PHP_VERSION,
                    'openssl'       => defined('OPENSSL_VERSION_TEXT') ? OPENSSL_VERSION_TEXT : null,
                    'curl_version'  => $curl['version'] ?? null,
                    'ssl_version'   => $curl['ssl_version'] ?? null,
                    'curl_cainfo'   => ini_get('curl.cainfo'),
                    'openssl_cafile'=> ini_get('openssl.cafile'),
                ],
                'dns' => ['api.videosdk.live' => $dns],
                'videosdk_https_probe' => [
                    'status' => $http->status(),
                    'ok'     => $http->ok(),
                    'body_snippet' => Str::limit($http->body(), 200),
                ],
            ];

            Log::info('videosdk.diagnostics', $data);
            return response()->json($data)->header('X-Correlation-ID', $cid);
        } catch (\Throwable $e) {
            Log::error('videosdk.diagnostics.error', ['cid' => $cid, 'error' => $e->getMessage()]);
            return response()->json(['error' => $e->getMessage(), 'correlation_id' => $cid], 500)
                ->header('X-Correlation-ID', $cid);
        }
    }
}
