<?php 
namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\SupportTicket;
use App\Models\Appointment;

class SupportController extends Controller
{
    // ✅ Get Appointments (only id & status)
    public function getSupportAppointment(Request $request)
    {
        // ✅ Validate request params           
        $validated = $request->validate([
            'patient_id' => 'required|exists:users,id',
        ]);

        $patientId = $validated['patient_id'];

        // ✅ Build Query (only select required columns)
        $appointments = Appointment::select('id as appointment_id', 'appointment_status', 'doctor_id')
            ->where('patient_id', $patientId)
            ->orderBy('id', 'DESC')
            ->get();

        return response()->json($appointments);
    }

    // Patient Support
    public function patientStore(Request $request)
    {
        $data = $request->validate([
            'patient_id'     => 'required|integer',
            'doctor_id'      => 'nullable|integer',
            'appointment_id' => 'nullable|integer', // ✅ new field
            'email'          => 'required|email',
            'subject'        => 'required|string|max:255',
            'message'        => 'required|string',
        ]);

        $data['doctor_id'] = $data['doctor_id'] ?? 0; // ✅ default 0
        $data['created_by'] = $data['patient_id'];
        $data['created_by_type'] = 'patient';

        $ticket = SupportTicket::create($data);

        return response()->json([
            'message'      => 'Support ticket created by patient',
            'ticket_id'    => $ticket->id,
            'ticket_no'    => $ticket->ticket_no,
            'appointment_id' => $ticket->appointment_id, // ✅ return appointment_id
            'status'       => "open",
        ], 200);
    }

    // Patient tickets
    public function patientTickets(Request $request)
    {
        $request->validate([
            'patient_id' => 'required|integer|exists:users,id'
        ]);

        $tickets = SupportTicket::where('patient_id', $request->patient_id)
            ->orderBy('created_at', 'desc')
            ->get(['id','ticket_no','appointment_id','subject','status','created_at']);

        return response()->json([
            'status' => true,
            'data'   => $tickets
        ]);
    }

    // ✅ Get Support Doctor Appointments
    public function getSupportDoctorAppointment(Request $request)
    {
        // ✅ Validate request params
        $validated = $request->validate([
            'doctor_id' => 'required|exists:users,id',
        ]);

        $doctorId = $validated['doctor_id'];

        // ✅ Build Query (only select required columns)
        $appointments = Appointment::select('id as appointment_id', 'appointment_status')
            ->where('doctor_id', $doctorId)
            ->orderBy('id', 'DESC')
            ->get();

        return response()->json($appointments);
    }

    // Doctor Support
    public function doctorStore(Request $request)
    {
        $data = $request->validate([
            'doctor_id'      => 'required|integer',
            'patient_id'     => 'nullable|integer',
            'appointment_id' => 'nullable|integer', // ✅ नया field
            'email'          => 'required|email',
            'subject'        => 'required|string|max:255',
            'message'        => 'required|string',
        ]);

        $data['created_by'] = $data['doctor_id'];
        $data['created_by_type'] = 'doctor';

        $ticket = SupportTicket::create($data);

        return response()->json([
            'message'   => 'Support ticket created by doctor',
            'ticket_id' => $ticket->id,
            'ticket_no' => $ticket->ticket_no,
            'appointment_id' => $ticket->appointment_id, // ✅ response में भी भेजा
            'status'    => "open",
        ], 200);
    }

    // Doctor tickets
    public function doctorTickets(Request $request)
    {
        $request->validate([
            'doctor_id' => 'required|integer|exists:users,id'
        ]);

        $tickets = SupportTicket::where('doctor_id', $request->doctor_id)
            ->orderBy('created_at', 'desc')
            ->get(['id','ticket_no','appointment_id','subject','status','created_at']);

        return response()->json([
            'status' => true,
            'data'   => $tickets
        ]);
    }
}
