<?php

namespace App\Http\Controllers\Patient;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Clinic;
use App\Models\PatientClinic;
use Illuminate\Support\Facades\Storage;

use App\Models\User;
use App\Models\Doctor;
use App\Models\DoctorProfile;
use App\Models\DoctorEducation;
use App\Models\DoctorExperience;
use App\Models\DoctorAvailability;
use App\Models\DoctorClinic;


class DoctorController extends Controller
{
    /**
     * Show the list of clinics/doctors.
     */
    public function index()
    {
        // Fetch doctors or clinics list
        // Example: $doctors = Doctor::with('clinic')->get();

        return view('patient.doctor.doctor-list'); // Blade: resources/views/patient/doctors/list.blade.php
    }

    /**
     * Add clinics/doctors.
     */
    public function store(Request $request)
    {
        // ✅ Validate
        // $request->validate([
        //     'clinic_id' => 'required|exists:clinics,id',
        // ]);

        $request->validate([
            'clinic_id' => 'required|exists:clinics,id',
        ], [
            'clinic_id.required' => 'Clinic ID is required.',
            'clinic_id.exists'   => 'You have entered an invalid Clinic ID.',
        ]);

        $patientId = Auth::id();
        $clinicId  = $request->clinic_id;

        // Check if already added
        $exists = PatientClinic::where('patient_id', $patientId)
            ->where('clinic_id', $clinicId)
            ->first();

        if ($exists) {
            return redirect()->route('patient.doctor.clinicList')
                ->with('error', 'Clinic already added for this patient.');
        }

        // $patientId = Auth::id(); // ✅ login patient detect
        // $clinicId  = $request->clinic_id;

        // // ✅ Check if already added
        // $exists = PatientClinic::where('patient_id', $patientId)
        //     ->where('clinic_id', $clinicId)
        //     ->first();

        // if ($exists) {
        //     return redirect()->back()->with('error', 'Clinic already added for this patient.');
        // }

        // ✅ Clinic ka doctor_id nikal lo
        $clinic = Clinic::find($clinicId);
        $doctorId = $clinic ? $clinic->owner_id : null;

        // ✅ Save record
        $patientClinic = PatientClinic::create([
            'patient_id'      => $patientId,
            'clinic_id'       => $clinicId,
            'doctor_id'       => $doctorId,
            'added_by'        => $patientId,
            'status'          => 'active',
            'approval_status' => 'approved',
            'notes'           => $request->notes ?? null,
        ]);

        // ✅ Redirect to clinic list with success flash message
        return redirect()->route('patient.doctor.clinicList')
            ->with('success', 'Clinic added successfully.');
        //return redirect()->back()->with('success', 'Clinic added successfully.');
    }

    /**
     * Show the list of clinicslist.
     */
    public function cliniclist()
    {
        $patientId = Auth::id(); // ✅ login patient detect

        // Patient clinics + related clinic details
        $clinics = PatientClinic::with('clinic')
            ->where('patient_id', $patientId)
            ->get();

        // अगर कोई clinic नहीं है
        if ($clinics->isEmpty()) {
            return view('patient.doctor.clinic-list', [
                'clinics' => [],
                'message' => 'No clinics found for this patient.',
            ]);
        }

        // map करके data तैयार करो
        $data = $clinics->map(function ($pc) {
            return [
                'patient_id'      => $pc->patient_id,
                'clinic_id'       => $pc->clinic_id,
                'doctor_id'       => $pc->doctor_id,
                'status'          => $pc->status,
                'approval_status' => $pc->approval_status,
                'notes'           => $pc->notes,
                'name'            => $pc->clinic->name,
                'email'           => $pc->clinic->email,
                'address'         => trim(
                    $pc->clinic->line1 . ', ' .
                    $pc->clinic->line2 . ', ' .
                    $pc->clinic->city . ', ' .
                    $pc->clinic->state . ', ' .
                    $pc->clinic->country . ' - ' .
                    $pc->clinic->pincode,
                    ', - '
                ),
                // 'specialities'    => $pc->clinic->specialities,
                // 'contact_numbers' => $pc->clinic->contact_numbers,

                'specialities'    => is_array($pc->clinic->specialities) 
                        ? implode(', ', $pc->clinic->specialities) 
                        : $pc->clinic->specialities,

                'contact_numbers' => is_array($pc->clinic->contact_numbers) 
                                        ? implode(', ', $pc->clinic->contact_numbers) 
                                        : $pc->clinic->contact_numbers,


                'rating'          => '0.0',
                'image'           => $pc->clinic->image 
                                        ? url('storage/' . $pc->clinic->image) 
                                        : asset('assets/img/hospital.png'),
            ];
        });

        return view('patient.doctor.clinic-list', [
            'clinics' => $data,
            'message' => null,
        ]);
    }

    /**
     * Show the list of clinicsdetail.
     */
    public function clinicdetail($id)
    {
        $patientId = Auth::id(); // ✅ current login patient

        // ✅ Check if clinic assigned to patient
        $relation = PatientClinic::where('clinic_id', $id)
            ->where('patient_id', $patientId)
            ->first();

        if (!$relation) {
            return redirect()->route('patient.doctor.clinicList')
                ->with('error', 'This clinic is not assigned to you.');
        }

        // ✅ Fetch clinic with doctors
        $clinic = Clinic::with('doctors')->findOrFail($id);

        $rating = '0.0';

        // ✅ Convert specialities & contact numbers to array
        $specialities = is_string($clinic->specialities)
            ? explode(',', $clinic->specialities)
            : (is_array($clinic->specialities) ? $clinic->specialities : []);

        $contactNumbers = is_string($clinic->contact_numbers)
            ? explode(',', $clinic->contact_numbers)
            : (is_array($clinic->contact_numbers) ? $clinic->contact_numbers : []);

        // ✅ Prepare doctors list with proper speciality & image
        $doctors = $clinic->doctors->map(function ($doctorRelation) {
            // Fetch fresh doctor record from doctors table
            $doctor = \App\Models\Doctor::find($doctorRelation->id);

            $user = \App\Models\User::find($doctor->user_id);

            // Prepare specialization array
            $specializations = [];
            if (!empty($doctor->specialization)) {
                $specializations = is_array($doctor->specialization) 
                    ? $doctor->specialization 
                    : explode(',', $doctor->specialization);
            } else {
                $specializations = ['---'];
            }

            // Prepare profile image
            $profileImage = $doctor->profile_image
                ? url('storage/'.$doctor->profile_image)
                : asset('assets/img/icons/sidebar-icon.svg');

            return [
                'id'             => $doctor->id,
                'name'           => $user->name ?? 'N/A',
                'email'          => $user->email ?? 'N/A',
                'phone'          => $user->phone_number ?? 'N/A',
                'specializations'=> $specializations,
                'profile_image'  => $profileImage,
            ];
        });

        //dd($doctors); // This will show all doctor objects with URLs

        return view('patient.doctor.clinic-detail', compact(
            'clinic',
            'rating',
            'specialities',
            'contactNumbers',
            'doctors'
        ));
    }

    /**
     * Show the list of doctordetail.
     */
    public function doctordetail($id)
    {
        // ✅ Fetch base user info
        $user = User::findOrFail($id);

        // ✅ Doctor profile info
        $doctorProfile = DoctorProfile::where('user_id', $id)->first();
        $doctor = Doctor::where('user_id', $id)->first();

        // ✅ Education & Experience
        $education = DoctorEducation::where('doctor_profile_id', optional($doctorProfile)->id)->get();
        $experiences = DoctorExperience::where('doctor_profile_id', optional($doctorProfile)->id)->get();

        // ✅ Availability slots
        $availabilities = DoctorAvailability::where('doctor_id', $id)->get();

        // ✅ Clinics doctor belongs to
        $clinics = DoctorClinic::where('doctor_id', $id)->get();

        // ✅ Assemble data for Blade view
        $data = [
            'doctor_id'         => $user->id,
            'name'              => $user->name,
            'email'             => $user->email,
            'phone'             => $user->phone_number ?? null,
            'status'            => $user->status,
            'specialization'    => $doctor->specialization ?? null,
            'bio'               => $doctor->bio ?? $doctorProfile->bio ?? null,
            'qualifications'    => $doctor->qualifications ?? null,
            'experience_years'  => $doctor->experience_years ?? null,
            'license_number'    => $doctor->license_number ?? null,
            'profile_image'     => $doctor->profile_image ? url('storage/' . $doctor->profile_image) : null,
            'rating'            => $doctorProfile->rating ?? null,
            'reviews_count'     => $doctorProfile->reviews_count ?? 0,
            'verification_status'=> $doctorProfile->verification_status ?? null,
            'education'         => $education,
            'experiences'       => $experiences,
            'availability'      => $availabilities,
            'clinics'           => $clinics,
        ];

        return view('patient.doctor.doctor-detail', compact('data'));
    }

    /**
     * Show the list of prescriptions for the patient.
     */
    public function prescriptions()
    {
        // Fetch prescriptions for logged-in patient
        // Example: $prescriptions = Prescription::where('patient_id', auth()->id())->get();

        return view('patient.doctor.prescriptions'); // Blade: resources/views/patient/prescriptions/list.blade.php
    }
}
