<?php

namespace App\Http\Controllers;
use App\Models\DoctorClinic;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

use Illuminate\Http\Request;

class AvailabilityController extends Controller
{
    // public function index()
    // {
    //     $availabilities = DB::table('doctor_availability')
    //     ->join('users', 'users.id', '=', 'doctor_availability.doctor_id')        // doctor_id points to users.id
    //     ->leftJoin('doctors', 'doctors.user_id', '=', 'users.id')                // profile comes from doctors
    //     ->select(
    //         'doctor_availability.*',
    //         'users.name as doctor_name',
    //         'users.email as doctor_email',
    //         'doctors.profile_image as doctor_image',
    //         'doctors.specialization',
    //         'doctors.license_number'
    //     )
    //     ->get();

    //     return view('doctor.availability.availability-list', compact('availabilities'));
    // }

    public function index()
    {
        $doctorId = auth()->id(); // Get logged-in doctor ID

        // Get clinic IDs associated with the doctor (primary/sub doctor)
        $clinicIds = DB::table('doctor_clinic')
            ->where('doctor_id', $doctorId)
            ->pluck('clinic_id');

        // Fetch availabilities only for clinics where doctor is associated
        $availabilities = DB::table('doctor_availability')
            ->join('users', 'users.id', '=', 'doctor_availability.doctor_id')
            ->leftJoin('doctors', 'doctors.user_id', '=', 'users.id')
            ->whereIn('doctor_availability.clinic_id', $clinicIds)
            ->select(
                'doctor_availability.*',
                'users.name as doctor_name',
                'users.email as doctor_email',
                'doctors.profile_image as doctor_image',
                'doctors.specialization',
                'doctors.license_number'
            )
            ->get();

        return view('doctor.availability.availability-list', compact('availabilities'));
    }


    public function create()
    {
        $authUser = auth()->guard('doctor')->user();

        if (!$authUser->hasRole('primary_doctor') || !$authUser->hasPrimaryClinic()) {
            return view('errors.403');
        }

        $clinicId = $authUser->primaryClinics[0]->id;

        $doctors = DoctorClinic::with(['doctor', 'clinic'])
            ->where('clinic_id', $clinicId)
            ->whereHas('doctor')
            ->get();

        return view('doctor.availability.availability-form', compact('doctors'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'clinic_id'     => 'required|integer',
            'doctor_id'     => 'required|integer',
            'slot_duration' => 'required|integer',
            'from_time'     => 'required|date_format:H:i',
            'to_time'       => 'required|date_format:H:i|after:from_time',
            'in_person_fee' => 'required|numeric|min:0',
            'video_fee'     => 'required|numeric|min:0',
            'days'          => 'required|array',
        ]);

        // Check if availability already exists for this doctor and clinic
        $existing = DB::table('doctor_availability')
            ->where('clinic_id', $validated['clinic_id'])
            ->where('doctor_id', $validated['doctor_id'])
            ->exists();

        if ($existing) {
            return back()->with('error', 'Availability for this doctor already exists. Please delete it before creating a new one.');
        }

        $from = Carbon::createFromFormat('H:i', $validated['from_time']);
        $to   = Carbon::createFromFormat('H:i', $validated['to_time']);
        $step = (int) $validated['slot_duration'];

        $slots = [];

        while ($from->copy()->addMinutes($step)->lte($to)) {
            $slotStart = $from->format('H:i');
            $slotEnd   = $from->copy()->addMinutes($step)->format('H:i');
            $slots[]   = "$slotStart - $slotEnd";
            $from->addMinutes($step);
        }

        DB::table('doctor_availability')->insert([
            'clinic_id'         => $validated['clinic_id'],
            'doctor_id'         => $validated['doctor_id'],
            'slot_duration'     => $validated['slot_duration'],
            'day'               => json_encode($validated['days']),
            'from_time'         => $validated['from_time'],
            'to_time'           => $validated['to_time'],
            'time_slot'         => json_encode($slots),
            'booked_time_slot'  => json_encode([]),
            'in_person_fee'     => $validated['in_person_fee'], // 💰 New
            'video_fee'         => $validated['video_fee'],     // 💰 New
            'status'            => '1',
            'other'             => '',
            'created_at'        => now(),
            'updated_at'        => now(),
        ]);

        return back()->with('success', 'Availability created successfully!');
    }

    public function destroy($id)
    {
        DB::table('doctor_availability')->where('id', $id)->delete();
        return back()->with('success', 'Availability deleted.');
    }

    public function toggleStatus($id)
    {
        $availability = DB::table('doctor_availability')->where('id', $id)->first();
        $newStatus = $availability->status == 1 ? 0 : 1;
        DB::table('doctor_availability')->where('id', $id)->update(['status' => $newStatus]);
        return back()->with('success', 'Status updated.');
    }
    
    public function edit($id)
    {
        $availability = DB::table('doctor_availability')->where('id', $id)->first();
        $doctors = DoctorClinic::with(['doctor', 'clinic'])->get(); // Assuming relationship is set

        return view('doctor.availability.availability-edit', compact('availability', 'doctors'));
    }

    public function update(Request $request, $id)
    {
        // Allow both H:i and H:i:s formats
        $validated = $request->validate([
            'clinic_id'     => 'required|integer',
            'doctor_id'     => 'required|integer',
            'slot_duration' => 'required|integer',
            'from_time'     => 'required|date_format:H:i',
            'to_time'       => 'required|date_format:H:i|after:from_time',
            'in_person_fee' => 'required|numeric|min:0',
            'video_fee'     => 'required|numeric|min:0',
            'days'          => 'required|array',
        ]);

        // Create time slots
        $from = Carbon::createFromFormat('H:i', $validated['from_time']);
        $to   = Carbon::createFromFormat('H:i', $validated['to_time']);
        
        $step = (int) $validated['slot_duration'];

        $slots = [];
        while ($from->copy()->addMinutes($step)->lte($to)) { 
            $slotStart = $from->format('h:i A');
            $slotEnd   = $from->copy()->addMinutes($step)->format('h:i A');
            $slots[]   = "$slotStart - $slotEnd";
            $from->addMinutes($step);
        }

        // Update DB
        DB::table('doctor_availability')->where('id', $id)->update([
            'clinic_id'         => $validated['clinic_id'],
            'doctor_id'         => $validated['doctor_id'],
            'slot_duration'     => $validated['slot_duration'],
            'day'               => json_encode($validated['days']),
            'from_time'         => $validated['from_time'],
            'to_time'           => $validated['to_time'],
            'time_slot'         => json_encode($slots),
            'in_person_fee'     => $validated['in_person_fee'], // 💰 New
            'video_fee'         => $validated['video_fee'],     // 💰 New
            'updated_at'        => now(),
        ]);

        return back()->with('success', 'Availability updated successfully!');
        //return redirect()->route('doctor.availability-list')->with('success', 'Availability updated successfully!');
        //return view('doctor.availability.availability-list')->with('success', 'Availability updated successfully!');
    }

}
